function rootPlotLab02SS(Gamma, Beta, Kp0, Kp1, Kd0, Kd1, Ki0, Ki1)
% Plot Step responses and natural frequencies for propeller arm 
% as Kp varies from Kp0->Kp1, (Kp0 > 0, Kp1 >= Kp0)
% Gamma: motor command scale factor (> 0).  
% Beta: Friction coefficient (<= 0). 
% The nat freqs corresponding to Kp1 (maximum Kp) marked with black star

% Optionally specify that Kd varies from Kd0->Kd,
if nargin < 5; Kd0 = 0.0; Kd1 = 0.0; end
% Optionally specify that Ki varies from Ki0->Ki1.  
if nargin < 7; Ki0 = 0.0; Ki1 = 0.0; end

% Auto rescaling around 1.
zoom = true;   % Zooms complex plane

% Should not need to change these
order = 2;  % 2:P; 3:PD; 4:PD,non-instant thrust; 5:PID, non-instant
if(Kd1 > 0); order = 3; end
if(Beta < 0); order = 4; end
if(Ki1 > 0); order = 5; end
dT = 1.0e-3;    % Time between samples
seconds = 5;    % Response plot time interval
nParamVals = 1000;  % Number of param value points

% Checks on input values
close all;
assert((Kd1 == 0 && order == 2) || (Kd1 > 0 && order > 2), "Order too low for nonzero Kd")
assert(((Beta == 0 && order < 4) || (Beta < 0 && order > 3)),"mismatch Beta and Order");
assert((Ki1 == 0 && order < 5) || (Ki1 > 0 && order > 4), "mismatch Ki and Order");

    % Function that computes system matrix A and eig(A)=natfreqs
    function [natFreqs,A,B] = getNatFreqs(Kp,Kd,Ki,G,B)
        % Gets A matrix and nat freqs, assumes first state is theta
        if order <= 2 
            A(1,:) = [1 dT];        % row one 
            A(2,:)  = [-G*dT*Kp 1]; % row two
        elseif order == 3
            A(1,:) = [1 0 dT];                   % row one 
            A(2,:) = [1 0 0];                    % row two 
            A(3,:) = [-G*(dT*Kp + Kd) G*Kd 1];   % row three
        else % includes non-instant thrust model
            A41 = 0.0;  %** FIX THIS!!     
            A42 = 0.0;  %** FIX THIS!!      
            A43 = 0.0;                    
            A44 = 0.0;  %** FIX THIS!!      
            if order == 4 
                A(1,:) = [1 0 dT 0];        % row one 
                A(2,:) = [1 0 0 0];         % row two
                A(3,:) = [0 0 1 dT];        % row three
                A(4,:) = [A41 A42 A43 A44]; % row four
            else % includes integral feedback
                A45 = 0.0;   %** FIX THIS!!
                A(1,:) = [1 0 dT 0 0];          % row one
                A(2,:) = [1 0 0 0 0];           % row two
                A(3,:) = [0 0 1 dT 0];          % row three
                A(4,:) = [A41 A42 A43 A44 A45]; % row four
                A(5,:) = [-dT 0 0 0 1];          % row five
            end
        end
        natFreqs = eig(A);
        B = B';
    end

[natFreqs,A0,B0] = getNatFreqs(Kp0,Kd0,Ki0,Gamma,Beta);
[~,A1,B1] = getNatFreqs(Kp1,Kd1,Ki1,Gamma,Beta);

% Preallocate vectors for time and state
T = 0:ceil(seconds/dT); 
X = zeros(size(A0,1),length(T));
X(1,1) = 1;  % Set initial theta to 1, other initial states to zero.

figure(1)
% Plot Step Responses
for plt = 1:2
    if plt == 1
        A = A0; Kp = Kp0; Kd = Kd0; Ki = Ki0;
    else 
        A = A1; Kp = Kp1; Kd = Kd1; Ki = Ki1;
    end
    for i = 2:length(T)
        X(:,i) = A*X(:,i-1);  % Assumes input is u[n] =1 for all n
    end
    subplot(3,1,plt);
    plot(T,X(1,:));
    xlabel('Time in Samples');
    title(['ZIR',',Kp=',num2str(Kp),',Kd=',num2str(Kd),',Ki=',num2str(Ki)]);
end


% Get an array # natural frequency rows, # param values column
natMat = zeros(size(natFreqs,1),nParamVals);
for i = 1:nParamVals
    Kdl = Kd0 + (Kd1-Kd0)*i/nParamVals;
    Kpl = Kp0 + (Kp1-Kp0)*i/nParamVals;
    Kil = Ki0 + (Ki1-Ki0)*i/nParamVals;
    natMat(:,i) = getNatFreqs(Kpl,Kdl,Kil,Gamma,Beta);
end

% Get slower (nearer |1|) nat freqs
if zoom
    zoomThresh = 0.95; 
else 
    zoomThresh = 0.0; 
end
rootsLarge = natMat(abs(natMat) >= zoomThresh);  % Get slower nat freqs

% Set the subplot
subplot(3,1,3); hold on;

% Plot trajectories of slower natural frequencies
for rootType = 1:3
    if(rootType == 1) 
        roots = rootsLarge(imag(rootsLarge) == 0); pStr = 'mo';
    elseif(rootType == 2) 
        roots = rootsLarge(imag(rootsLarge) > 0); pStr = 'ro';
    else 
        roots = rootsLarge(imag(rootsLarge) < 0); pStr = 'go';
    end
    plot(real(roots), imag(roots),pStr); 
end
if length(natMat) > 4
   last4 = natMat(:,end-4:end); 
   last4 = last4(abs(last4) >= zoomThresh);
   plot(real(last4), imag(last4),'k*');
end

% Plot entire (or fraction of if zoom true) of the unit circle
if zoom 
    maxAngle = max(abs(angle(rootsLarge(:)))); 
else
    maxAngle = pi;
end
ang = linspace(-1.5*maxAngle, 1.5*maxAngle, 10000);
circlepts = cos(ang)+ 1j*sin(ang);
plot(real(circlepts), imag(circlepts), 'b.');
hold off;

% Give the set of plots a title
title(['unitCirc(Blue)',...  
        ', Gam=',num2str(Gamma), ', Beta=', num2str(Beta),...
        ', Kp:',num2str(Kp0),'->',num2str(Kp1),...
        ', Kd:',num2str(Kd0),'->',num2str(Kd1),...
        ', Ki:',num2str(Ki0),'->',num2str(Ki1)]);
end

