% This script:
% 1) Runs labFitter to convert frequency sweep data into a state-space model.  
% 2) SS model used to design CT and DT measured-state- and 
%    observed-state-feedback (Msf and Obsf) controllers.  
% 3) The CT Msf and Obsf controller phase margins are compared.
% 4) The DT Msf and Obsf controller simulations are compared 
%    (with noise and disturbances).
% 5) The observer-based controller and associated gains are written to 
%    file includable in appropriate Teensy sketch.
%

close all; % close all the plots.
clearvars;
dt = 1000e-6;        % DT sample period (1000 samples per second)
doIntegral = false;  % Set to true if adding error integral as a state.
plotSimCt = false;   % Plot the CT margins and time response if needed.

% Number of poles and zeros for the fitter to use.
nPoles = 3;  % ******* Why Three Poles?
nZeros = 0;  % ******* Why no zeros?

% Simulation Inputs
inAmp = 0.15;             % Input Amplitude for Simulation.

% Noise Inputs
noiseVec = [2 100 10000]';   % Noise vector, How is this being used?
noiseAmp = 0.0*0.005;    % Noise amplitude (set by experiment)

% Disturbance Inputs 
disturbScaleVec= [1,100,1000]; % Scale factors for position, vel, accel.
disturbEqn = 1;  % *******CHANGE THIS FOR TESTING********!!
disturbAmp = 0.1;
disturbAmp= disturbAmp*disturbScaleVec(disturbEqn);

% Createfrequency sweep data matrix M
% M: real, five columns and num rows = num measurements;
% M: Real, row per measurement, 5 Cols: Freq,re(U),imag(U),re(Y),imag(Y)
% M = synthData(0.25, 5, 40); (uncomnment for testing)
% M from File, Format: Line = row in M, comma-sep-vals with semicolon EOL.
M = readmatrix('sweep.csv', 'LineEnding', ';'); 

% Fit state-space model to frequency sweep data
sysOpenLoop = labFitter(nPoles,nZeros,M);
A = sysOpenLoop.a; 
B = sysOpenLoop.b; 
C = sysOpenLoop.c; 
D = sysOpenLoop.d;

% Find the gains for a measured-state and estimated-state DT controller
% Examine state-space A, B, C matrices to determine what weights to use.
Q = diag([100, 1, 1]);           % ********OPTIMIZE THIS***********                     
R = 1;                             % ********OPTIMIZE THIS***********

% Compute state-feedback gains using lqr.
K = lqr(A,B,Q,R);                     
Kr = -1/(C*((A-B*K)\B));
By = B*Kr;  % usually By=B*Kr, but not e.g. when using integrated error.

% Find gains for an observer using LQR
Ql = diag([0.1,0.1,0.1]);         % ********OPTIMIZE THIS***********
Rl = 0.01;                        % ********OPTIMIZE THIS***********
L = lqr(A',C',Ql, Rl)';  

disp('***CT Controller and Observer Gains********');
disp(['K: ',num2str(K), '     Kr: ', num2str(Kr)]);
disp(['L: ',num2str(L')]);
disp('***CT Control and Error-Estimate Poles**');
disp(['controller poles: ', num2str(eig(A-B*K).')]);
disp(['estimator poles: ', num2str(eig(A-L*C).')]);

% Create input matrices for two additional inputs, disturbance and noise.
Bdist = zeros(size(A,1),1);
Bdist(disturbEqn) = disturbAmp;
Bnoise = noiseAmp*noiseVec(1:nPoles);  % Relative noise for state measurements.

%%%%%%%%%%%%%%%%%% Switch to Discrete-Time ***********************

% Convert system to discrete time (exact for ZOH).
Ad = expm(A*dt);  % Matrix Exponential           
B2Bd = (Ad - eye(size(Ad)))/A; % Matrix for scaling B to Bd.
Bd = B2Bd*B;
inBdistd = B2Bd * Bdist; % Scale CT state disturbance for DT
Cd = C;
Dd = D;

% Recompute the controller gains (K's) and observer gains (L's) for DT.
Kd = dlqr(Ad, Bd, Q, R); % Kd should be same as K if sampling fast enough
Ld = dlqr(Ad',Cd',Ql,(1/dt^2)*Rl)'; % Rl scaled by samples/second squared.
%Ld = place(Ad',Cd',[0.97,0.98,0.99])';

% Calc closed-loop matrices.
if ~doIntegral
    Adcl = Ad - Bd*Kd;
    Adlc = Ad - Ld*Cd;
    Krd = 1/(Cd*((eye(size(Ad)) - Adcl)\Bd));
    Byd = Bd*Krd; % Not doing the integral.
else
    % Matrix row for extra state = DT-sum approximate to error integral
    nPoles = nPoles+1;
    Ad(nPoles,1:nPoles) = zeros(1,nPoles);   %*INTEGRAL FIX THIS!**
    Bd(nPoles) = 0;                             %*INTEGRAL FIX THIS!**
    Cd(nPoles) = 0;                             %*INTEGRAL FIX THIS!**
    Bnoise(nPoles) = 0;
    inBdistd(nPoles) = 0;

    % Find Kd for system with additional integrated error state.
    Qp = diag(zeros(1,nPoles));                     %*****INTEGRAL FIX THIS!****
    Rp = 1.0;
    Kd = dlqr(Ad, Bd, Qp, Rp);  
    Ld = zeros(nPoles,1);                            %*****INTEGRAL FIX THIS!****

    Adcl = Ad - Bd*Kd; % Closed loop system matrix.
    Adlc = Ad - Ld*Cd;                    %****INTEGRAL FIX THIS!! ***
    Krd = 0; % Fix THIS?
    Byd = zeros(nPoles,1);                %*****INTEGRAL FIX THIS!****
end

disp(' ');
disp('***DT Controller and Observer Gains********');
disp(['Kd Gains: ',num2str(Kd), '      Krd:',num2str(Krd)]);
disp(['Ld gains: ',num2str(Ld')]);
disp('*****||DT Poles||*****');
disp(['controller poles: ', num2str(sort(abs(eig(Adcl)).'))]);
disp(['estimator poles: ', num2str(sort(abs(eig(Adlc)).'))]);

% *****Simulate Measured-State and Observed-State Feedback (Msf,Obsf)

% Discrete time inputs 
inByd = inAmp*Byd;  
inKrd = inAmp*Krd;
[yMsf,yObsf,t,uDist] = ...
    simMsfObsf(Ad,Bd,Cd,Kd,inKrd,Ld,inByd,inBdistd,Bnoise,dt);

% Plot DT Measured and Observed State.
nPlots = size(yMsf,2);  % Measured-state and Observed-State, same num plots.
plotSys(t, yMsf, nPlots, nPoles, uDist, disturbEqn, 'DT Measured State Feedback');
plotSys(t, yObsf, nPlots, nPoles, uDist, disturbEqn, 'DT Observed State Feedback');

% Output gains and model to a file compiled by Teensyduino
fname = 'obs6310.h'; % If you change this name, change it in sketch!!!
printTeensyObserver(Ad,Bd,Cd,Krd,Kd,Ld,fname)

% Simulate and plot continous time responses if desired.
if (plotSimCt)
    % Open-loop margin plots for measured-state and observed-state feedback
    tfu2Kx = tf(ss(A,B,K,D));   % Open-Loop (u->Kx) transfer function
    [Gm,Pm,Wca,Wcb] = margin(tfu2Kx);  % u->Kx Margin plot.

    tfu2y = tf(ss(A,B,C,D));     % Open loop (u->y) transfer function
    tfy2ObsKx = ...              % Open loop (y->KxObs) transfer function
        tf(ss(A-(B*K + L*C), L, K, D));
    tfu2KxObs = tfu2y*tfy2ObsKx; % Open loop (u->KxObs) transfer function
    [GmObs, PmObs, WcaObs, WcbObs] ...
    = margin(tfu2KxObs);  % u->Kxobs Margin plot
    % Margin plots for measured-state-feedback and observer-state-feedback 
    % open-loop systems.
    figure;
    margin(tfu2Kx); hold on;
    margin(tfu2y*tfy2ObsKx); hold off;
    measString = sprintf('Meas(Blue) PM=%0.0fdeg(%0.0frads/s) ', Pm, Wcb);
    obsString = sprintf('Obs(red) PM=%0.0fdeg(%0.0frads/s)', PmObs,WcbObs);
    title([measString,obsString]);

    % Scale inputs for CT
    inBy = inAmp*By; %#ok<*UNRCH>
    inKr = inAmp*Kr;
    inBnoise = 0.0*Bnoise; % Now way to experimentally calibrate
    [yMsf,yObsf,t,uDist] = simMsfObsf(A,B,C,K,inKr,L,inBy,Bdist,inBnoise);
    nPlots = size(yMsf,2);  % Measured-state and Observed-State, same num plots.
    plotSys(t, yMsf, nPlots, nPoles, uDist, disturbEqn, 'CT Measured State Feedback');
    plotSys(t, yObsf, nPlots, nPoles, uDist, disturbEqn, 'CT Observed State Feedback');
end
 
