// ****************** MUST INCLUDE LINES BELOW*****************************
#include <lib6310.h>
// Init library and set DAC resolution, ADC resolution, serial channel
#define MYSERIAL Serial
lib6310 my6310(12,12,&Serial);
// Set up the analog-to-digital library, with a buffersize
#define sampleBufN 500
LIB6310_ADC(sampleBufN+100);  // ADC sample buffer of 100+sampleBufN
// ****************** MUST INCLUDE LINES ABOVE*******1**********************

// Constants for Users to Modify 
// ************************************************************
#define AngleOffset 0.0     // Adjust if measured angle is not zero when horizontal. 
#define nomCmd 0.35           // Nominal Command sets linearization
#define MODE 1          // -1:Sweeper Mode   0: Calibration Mode 1:Control Mode with Observer ()

// MODE -1: Save Sweep:         Sweeper generates Input, Output to file, use PD-like controller
// MODE  0: Calibrate Sensor:   Sweeper generates Input, Output to serial plotter, use PD-like controller
// MODE  1: Control W/Observer  User input, Output to serial plotter, use matlab generated observer control
#if MODE == 0  
  #define SRCTYPE 0             // 0:y_d=0 
  #define OUTTYPE 0             // 0:Wave Plots 1:FreqRes Plots 2:Freq Res Print (for fitter)
  #define UseGenFile false      // Set true to use observer design
#elif MODE == -1 
  #define SRCTYPE -1             // -1:Sweep y_d 
  #define OUTTYPE 2             // 0:Wave Plots 1:FreqRes Plots 2:Freq Res Print (for fitter)
  #define UseGenFile false      // Set true to use observer design
#else 
  #define SRCTYPE 1             // 1:User y_d 
  #define OUTTYPE 0             // 0:Wave Plots 1:FreqRes Plots 2:Freq Res Print (for fitter)
  #define UseGenFile true      // Set true to use observer design
#endif

#define controllerUpdatePeriod 1000  // Sampling 1000 times/second!!!

// Frequency Sweep Parameters
#define firstFreq 0.1
#define lastFreq 4
#define MAX_FREQ_PTS 30

// Indexes for values being sent from plotter send window. 
// e.g. to set the input period to two seconds, type "5 0.5" and click send.
// To set the input amplitude for a square wave, type e.g. "6 0.1", for a sine
// wave type "6 -0.1".
#define EXTRABACK 0  // dx/dt is approx (x[n]-x[n-m])/(DeltaT*m), m = 1+extraBack
#define KD 1         // Derivative Gain
#define KP 3         // Proportional Gain
#define SKIPS 9      // Number of sample points skipped between plotted points
#define FREQ 5       // Input frequency, only used in MODE 1
#define AMP 6        // Input amplitude for square wave (if > 0) or sine (if < 0)
#define EST 7        // Use estimated state if > 0, measured state if < 0.

// offsets for parameters sendable from plotter window. Set offsetEstState to 1.0 to default
// to using the observer to estimate states.  In that case, type "7 -1" to switch back to measured state
#define offsetExtraBack 20         // 20 is very high, used to reduced noise and make plots interpretable.
#define offsetKd 0.06              // Low gains for model fitting sweep
#define offsetKp 0.5              // Low gian for model fitting sweep
#define offsetNumSkips 4           // plot point every 5 samples, 5*2ms=10ms, or 100 points per second.
#define offsetFreq 0.25            // 1/5 period per second, transition every 2 seconds.
#define offsetAmp 0.15             // Input togggles between -0.15 and 0.15 radians.
#define offsetEstState 1           // Positive, so use estimated state

// Constants for advanced users to modify.
#define Psmooth 0.0  // 0.0: no smoothing  0.999: very smooth
#define MbackMax 250 // dMeas/dt est by (Meas[n]-Meas[n-Mback] )/(DeltaT*Mback)

// ************************************************************************
// End of Constants and code for Users to Modify 
// SEE END OF FILE FOR USER-ALTERABLE CONTROL LOOP!!

// ********* State-Space Variable Definitions *****************
typedef float vec3[3];  // Typedef for 3-length vectors.
typedef vec3 mat3x3[3]; // Typedef for 3x3 matrices

// SET THESE USING obs6310.h, changes here WILL BE OVERWRITTEN!!!!!!

// Controller (Kr,Ksum, and vector of state gains K) and Observer (L) gains.
float Kr = 0.0;         // y_d scale factor.
float Ksum = 0.0;       // Integral of error gain.
vec3 K = {0.0,0.0,0.0}; // Gains for theta(y), omega(dy/dt), and alpha(d^2y/dt^2)
vec3 Ld = { 0.0,0.0,0.0 }; // Observer gains.

// DT State-Space Matrices Ad,Bd,Cd,Ld, 
vec3 Bd = { 0,0,0 }; // Prop model.
vec3 Cd = { 0.0,   0.0,   0.0 };
mat3x3 Ad = {
  {0.0, 0.0, 0.0},
  {0.0, 0.0, 0.0},
  {0.0, 0.0, 0.0}
};

// *******SETTING UP THE MONITOR DISPLAY (SLIDERS AND PLOTS). *************
// Sliders or settings from plotter window.
String sliders = "S~extraBack~0~20~S~Kd~0~5~S~Ksum~0~100~S~Kp~0~10~S~Skips~0~50~S~F~-3~3~S~Amp~-3~3~S~Est~0~1~";
String measEstY= "P~CMD~-4.0~-2.0~500~1~P~Y=X1 YD YEst=X1Est~-2.0~2.0~500~3~";
//String measEstX0= "P~X0Est X0~-0.5~0.5~500~2~"; // X0 equals y.
String measEstX2= "P~X2/5 X2Est/5~-0.5~2.0~500~2~";
String measEstX3= "P~X3/10 X3Est/10~-2.0~2.0~500~2~";

String monPlots = measEstY + measEstX2 + measEstX3;


#define PLOT_CMD 0
#define PLOT_Y 1
#define PLOT_YD 2
#define PLOT_YEST 3
#define PLOT_X2 4
#define PLOT_X2EST 5
#define PLOT_X3 6
#define PLOT_X3EST 7
#define PLOT_INTERR 8

#define PLOT_F 0
#define PLOT_CMD_S 1
#define PLOT_CMD_C 2
#define PLOT_OUT_S 3
#define PLOT_OUT_C 4
#define PLOT_IN_S 5
#define PLOT_IN_C 6


// Configuration string with surrounding limits.
String config_message = sliders + monPlots;

// Control Loop Input Functions
// ************************************************************************

// Measures Angle in Radians
float getTheta() {
  uint16_t numSamp = 9;
  float theta = 0.0;
  for(int i = numSamp; i > 0; i--) {
    theta += 2*PI*(my6310.adcGetAvg(IN4) - 0.5);
  }
  theta = theta/float(numSamp) - AngleOffset;
  return(theta);
}

// Approximates d(x)/dt as (x[n] - x[n-m])/(m*deltaT)
float dXdt(float x, int m, float delT, float &dx2Dt2Val)
{
  static float xBack[MbackMax+1];  // Stores past errors for deriv approx

  m = max(m,1); // Make sure m is at least 1.

  // save past errors.
  for (int i = MbackMax; i > 0; i--) xBack[i] = xBack[i-1];
  xBack[0] = x;

  // Calculate approximation to d(x)/dt
  float dt = float(m)*delT;
  float dxDtVal = (x- xBack[m])/dt; 
  dx2Dt2Val = (x- 2.0*xBack[m] + xBack[2*m])/(dt*dt); 

  return(dxDtVal);
}

// Approximately integrates x with bounds.
float errSum = 0.0;  // Stores accumulated errors.
float lastKi = 0.0;  // Used to zero sum when Ki changes.
// Computes the deltaT weight sum of errors.
float integX(float x, float Ki, float delT)
{
  static float xSum = 0.0;
  static float lastKi = 0.0;

  if(Ki > 0) {
    if(lastKi != Ki) xSum = 0.0;
    float SumMax = 100/(0.1 + Ki);
    xSum += x * delT; 
    xSum = max(min(SumMax,xSum), -SumMax);
  }
  lastKi = Ki;
  return(xSum);
}

// Update input when using sweeper
#define deg2rad 0.0175
#define TSSS 6.0 // Max num seconds to steady-state
boolean Pause = false;

float updateSweeper(float cmd, float theta, float theta_d, float delT) {
  static int numDts = 0;
  static int periodDts = 0;
  static int sssDTs = 0;
  static int freqIndex = 0;
  static float Freq = 0.0;
  static float sinv = 0.0;
  static float cosv = 0.0;
  static float sc = 0.0; 
  static float cc = 0.0;
  static float sa = 0.0;
  static float ca = 0.0;
  static float sd = 0.0;
  static float cd = 0.0;
  static float Fmultiplier = exp(log(lastFreq/firstFreq)/float(MAX_FREQ_PTS));

  // Update projections uses sinv and cosv from last call.
  sc += cmd*sinv; cc += cmd*cosv; 
  sa += theta*sinv; ca += theta*cosv;
  sd += theta_d*sinv; cd += theta_d*cosv;
  
  if( ((numDts % periodDts) == 0) || (numDts >= sssDTs) ) { //period Bound
    if(numDts >= sssDTs) {  // Ready to save steady-state
      // Save mag and phase if requested.
      if (OUTTYPE == 1) {
        float loopStatus[7];
        // Divide by 1000 to undo library scaling
        float scaleSum  = sd*1000;
        loopStatus[PLOT_F] = Freq/1000; 
        loopStatus[PLOT_CMD_S] = sc/scaleSum;
        loopStatus[PLOT_CMD_C] = cc/scaleSum;
        loopStatus[PLOT_OUT_S] = sa/scaleSum;
        loopStatus[PLOT_OUT_C] = ca/scaleSum;
        loopStatus[PLOT_IN_S] = sd/scaleSum;
        loopStatus[PLOT_IN_C] = cd/scaleSum;
        my6310.sendStatus(5,loopStatus);
      } else if(OUTTYPE == 2) {
        Serial.print(Freq);
        Serial.print(",");
        Serial.print(sc/sd,6);
        Serial.print(",");
        Serial.print(cc/sd,6);
        Serial.print(",");
        Serial.print(sa/sd,6);
        Serial.print(",");
        Serial.print(ca/sd,6);
        Serial.println(" ");
      }

      // Update frequency
      if((freqIndex == 0) || (freqIndex > MAX_FREQ_PTS)) { 
        Freq = firstFreq;
        freqIndex = 0; 
      } 
      else Freq *= Fmultiplier;
      freqIndex += 1;

      // Tweak frequency to be exactly N deltaT's
      periodDts = 2*int(0.5/(delT*Freq));
      Freq = float(1.0)/(delT*float(periodDts)); 

      // Wait for SS for 3 periods or TSSS, whichever is larger.
      int oscPeriods = 1+int(TSSS/(delT*float(periodDts)));
      sssDTs = max(oscPeriods,3) * periodDts; // Wait at least 3 periods for SSS
      numDts = 0;
    }
    // Reset projection parameters at end of each period.
    sc = 0; cc = 0; sa = 0; ca = 0; sd = 0; cd = 0;
  } // End of period boundary case.

  numDts += 1;
  float arg = 2*PI*float(numDts)/float(periodDts);
  sinv = sin(arg);
  cosv = cos(arg);
  
  return(sinv);
}

// Get Theta in radians
elapsedMicros loopTimer = 0;  // Used to synchronize loop start
float deltaT = 0.0;

// Setup function
void setup() {
  my6310.setup(config_message);
  my6310.startPWM(0.98); // start parallel drive pwm, A+B in sync.
  my6310.setupADC(16,false);
  deltaT = float(controllerUpdatePeriod)*1.0e-6;

#if UseGenFile
  #include "obs6310.h"
#endif

}

//********************** Main USER ALTERABLE Control Loop **********************
//**************************************************************************
// Estimated and Measured state vectors
#define stateBound 500.0
vec3 xPred = { 0.0, 0.0, 0.0 };
float xSum = 0.0;

// Counts plot-skipped samples.
int skipCounter = 0;  
float ydNext = 0.0;
float headRoom = 0.0;

void loop() {  
  // Initialize and check for usb inputs.
  my6310.startloop(); 

  // Get send window parameters.
  int extraBack = offsetExtraBack + my6310.getSlider(EXTRABACK); 
  #if MODE < 1
  float Kd = offsetKd + my6310.getSlider(KD); 
  float Kp = offsetKp + my6310.getSlider(KP); 
  #endif
  int numSkipsIn = offsetNumSkips + my6310.getSlider(SKIPS);
  float freqIn =  offsetFreq + my6310.getSlider(FREQ);
  float ampIn = offsetAmp + my6310.getSlider(AMP);
  float estState = offsetEstState + my6310.getSlider(EST);

  // Determine Nominal left and right command tilt from potentiometer. 
  float nomCmdTilt = 0.25*nomCmd*(my6310.analog2param(IN8) - 0.5);
  float nomCmdL = nomCmd + nomCmdTilt;
  float nomCmdR = nomCmd - nomCmdTilt;
  float nomMax = nomCmd+abs(nomCmdTilt); // Used for clipping command

  // Headroom is faction of period left until next sensor read.
  //float headRoom = float(controllerUpdatePeriod - int(loopTimer));

  // Wait exactly one deltaT since last sensor read.
  while (int(loopTimer) < controllerUpdatePeriod) {};
  loopTimer = 0;

  // Measure arm angle
  float y = getTheta();
 
  // Get the desired arm angle.
  float y_d = 0.0; 
  if(SRCTYPE == -1) y_d = ydNext;
  else if(SRCTYPE == 1) y_d = my6310.updateDesired(ampIn,freqIn,Psmooth);

  // Compute velocity (dyDt) and acceleration (dy^2/dt^2)
  float dy2Dt2 = 0.0;
  float dyDt = dXdt(y, 1+extraBack, deltaT, dy2Dt2); // Note: dy2Dt2 updated in call!

  // Updates running sum of errors (approx integral).
  // Check the signs of y and y_d below, make sure they are
  // consistent with model.
  float intErr = integX((y-y_d), Ksum, deltaT); 

  // Copy y, dy/dt, and dy2Dt2 into measured states
  vec3 xMeas;
  xMeas[0] = y;
  xMeas[1] = dyDt;
  xMeas[2] = dy2Dt2; 

  // Use xPred from last sample to compute state estimate xEst 
  // Copy and bound to prevent accumulation overflow.
  vec3 xEst;
  copy_bound_vec3(xEst,xPred,stateBound); 

  // Compute the feedback contribution to the command
#if MODE < 1  // Running sweeper, just use PD-like control.
    float u = Kp*(y_d-y) - Kd*dyDt;
#else // Running state-space controller, u = Kr*y_d - (K*x + Ksum*intErr)
    float u = Kr*y_d - Ksum*intErr; 
    if(estState > 0) u -= vec3_mul_inner(K,xEst);  
    else u -= vec3_mul_inner(K,xMeas);
#endif

  // Use the DT model to predict the state for next sample (xPred)
  // xPred = Ad*xEst + Bd*u + Ld*(yMeas - yPred), 
  vec3 t1, t2, t3, t4;
  float yEst = vec3_mul_inner(Cd,xEst);
  mat3x3_mul_vec3(t1, Ad, xEst); // t1 = Ad*xEst
  vec3_scale(t2,Bd,u); // t2 = Bd*u
  vec3_scale(t3, Ld, y-yEst); // t3 = Ld*(yMeas-yEst)
  vec3_add(t4,t2,t1); // t4 = t2+t1 = Ad*xEst+Bd*u
  vec3_add(xPred,t4,t3); // xPred = Ad*xEst+Bd*u+Ld*(yMeas-yEst)

  // Clip the command to ensure left or right 0 <= nomCmdL/R +/- u <=1
  // This CODE IS IN THE WRONG PLACE ********FIX THIS*****************
  float nomClip = max(nomMax,1.0-nomMax);
  u = my6310.scaleAndBound(u, nomClip, 1.0); 

  // Write the command to the motors.
  my6310.hbridgeBipolar(nomCmdL+u, B2L,B1L); // B1L <-> B2L swaps direction
  my6310.hbridgeBipolar(nomCmdL+u, A2L,A1L); // A1L <-> A2L swaps direction
  my6310.hbridgeBipolar(nomCmdR-u, A2R,A1R); // A1R <-> A2R swaps direction
  my6310.hbridgeBipolar(nomCmdR-u, B2R,B1R); // B2R <-> B1R swaps direction

  if(SRCTYPE == -1) ydNext = ampIn*updateSweeper(u, y, y_d, deltaT);

  if(skipCounter > 0) skipCounter--;
  else {  // Send data to monitor.
    skipCounter = numSkipsIn;
    if (OUTTYPE == 0) {
      float loopStatus[10];
      loopStatus[PLOT_Y] = y;   // headRoom;
      loopStatus[PLOT_YD] = y_d;
      loopStatus[PLOT_YEST] = yEst*estState;
      loopStatus[PLOT_CMD] = u - 3.0;
      loopStatus[PLOT_X2EST] = xEst[1]/5.0*estState;
      loopStatus[PLOT_X2] = xMeas[1]/5.0;
      loopStatus[PLOT_X3EST] = (xEst[2]/10.0)*estState;
      loopStatus[PLOT_X3] = xMeas[2]/10.0;
      loopStatus[PLOT_INTERR] = intErr;
      my6310.sendStatus(8,loopStatus);  
    }
  }
}




  


  
