// ****************** MUST INCLUDE LINES BELOW*****************************
#include <lib6310.h>
// Init library and set DAC resolution, ADC resolution, serial channel
#define MYSERIAL Serial
lib6310 my6310(12,12,&Serial);
// Set up the analog-to-digital library, with a buffersize
#define sampleBufN 500
LIB6310_ADC(sampleBufN+100);  // ADC sample buffer of 100+sampleBufN
// ****************** MUST INCLUDE LINES ABOVE*******1**********************

// Constants for Users to Modify 
// ************************************************************
#define AngleOffset 0.0 // Adjust so measured angle=0 when horizontal. 
#define propOffset 0.0 // half to right prop -half to left prop to level arm
#define Psmooth 0.0 // Toggle smoothing 0: none, 0.9->0.999: less->more smooth
#define DeltaT_Microsecs 1000 // Update period in microseconds

// offsets for parameters sendable from plotter window.
#define offsetExtraBack 2
#define offsetKd 0.0
#define offsetKi 0.0
#define offsetKp 0.0
#define offsetNumSkips 9
#define offsetFreq 0.0
#define offsetAmp 0.0

// ************************************************************************
// End of Constants and code for Users to Modify 
// SEE END OF FILE FOR USER-ALTERABLE CONTROL LOOP!!

// ***********NEVER MODIFY THESE INDICES*******************
// Indexes for values being sent from plotter send window.
// e.g. Set Kp by typing "3 0.3" in send window and clicking send.
#define EXTRABACK 0 // deriv ~= (Meas[n]-Meas[n-(1+EXTRABACK)])/(DeltaT*(1+EXTRABACK))
#define KD 1
#define KI 2
#define KP 3
#define SKIPS 4  // Number of sample points skipped between plotted points
#define FREQ 5 // Input frequency 
#define AMP 6  // Input peak amplitude, Amp>0:step inputs, Amp<0:sine inputs
// ***********NEVER MODIFY THE ABOVE INDICES*******************

#define MbackMax 300  // Maximum history of backward points

// *******SETTING UP THE MONITOR DISPLAY (SLIDERS AND PLOTS). *************
// Sliders or settings from plotter window.
String sliders = "S~ExtraBack~0~20~S~Kd~0~5~S~Ki~0~100~S~Kp~0~10~S~Skips~0~50~S~Freq~0.1~20~S~Amp~-3~3~";

// Plots: USE SPACES BETWEEN LABELS FOR MULTIPLOTS!!
String measDesPlots = "P~Theta_m Theta_d Cmd~-1~1~500~3~"; 
String errPlots = "P~DEdt/10  ErrSum~-1~1~500~2~";
#define PLOT_TM 0
#define PLOT_TD 1
#define PLOT_CMD 2
#define PLOT_DDT 3
#define PLOT_SUM 4

// Configuration string with surrounding limits.
String monPlots = measDesPlots + errPlots; 
String config_message = sliders + monPlots;


// Control loop setup
// ************************************************************************

// Motor PWM and Sensor defines.
elapsedMicros loopTimer = 0;  // Loop runs at least 10xsecond
float deltaT = 0.0;
float headRoom;

void setup() {
  my6310.setup(config_message);
  my6310.startPWM(1.0); // start parallel drive pwm, A+B in sync.
  my6310.setupADC(16,false);
  loopTimer = 0;
  deltaT = float(DeltaT_Microsecs)*1.0e-6;
  headRoom = DeltaT_Microsecs;
}

//********************** Main USER ALTERABLE Control Loop **********************
//**************************************************************************
int skipCounter = 0;
float pastError[MbackMax+1];  // Stores past errors for deriv approx
float pastMeas[MbackMax+1]; // Stores past angle values
float errSum = 0.0;  // Stores accumulated errors.
float lastKi = 0.0;

void loop() {  

  my6310.startloop(); // Initialize and check for usb inputs.

  // Get the nominal propeller command.
  float nomCmdIn = my6310.analog2param(IN8);

  // Get send window parameters.
  int extraBackIn = offsetExtraBack + my6310.getSlider(EXTRABACK); 
  float Kd = offsetKd + my6310.getSlider(KD);
  float Ki = offsetKi + my6310.getSlider(KI);
  float Kp = offsetKp + my6310.getSlider(KP); 
  int numSkipsIn = offsetNumSkips + my6310.getSlider(SKIPS);
  float freqIn =  offsetFreq + my6310.getSlider(FREQ);
  float ampIn = offsetAmp + my6310.getSlider(AMP);
  int mBack = 1 + extraBackIn;

  // Headroom is faction of period left until next sensor read.
  float headRoomThis = float(DeltaT_Microsecs - int(loopTimer));
  headRoom = headRoomThis;

  // Wait exactly one deltaT since last sensor read.
  while (int(loopTimer) < DeltaT_Microsecs) {};
  loopTimer = 0;
  
  // Measure angle and get desired angle and compute error.
  uint16_t numSamples = 12;
  float meas = 2*PI*(my6310.adcGetAvg(IN4, numSamples) - 0.5) - AngleOffset;
  float desired = my6310.updateDesired(ampIn,freqIn,Psmooth);   
  float error = (desired - meas);

  // save past errors.
  for (int i = mBack; i > 0; i--) {
    pastError[i] = pastError[i-1];
    pastMeas[i] = pastMeas[i-1];
  }
  pastError[0] = error;
  pastMeas[0] = meas;
  
  // Compute approximate d(Meas)/dt and d(error)/dt
  float mDeltaT =  float(mBack)*deltaT;  
  float dMeasDt = meas - pastMeas[mBack];
  float dErrDt = error - pastError[mBack];
  #define BV 200.0
  dMeasDt = my6310.scaleAndBound(dMeasDt, BV, mDeltaT);
  dErrDt = my6310.scaleAndBound(dErrDt, BV, mDeltaT);

  // Compute Integral Approximation.
  float SumMax = 100/(0.05+Ki);
  if(Ki != lastKi) { errSum = 0; lastKi=Ki; } // Zero integral if change in Ki.
  errSum = my6310.scaleAndBound(errSum+deltaT*error, SumMax, 1.0);

  // Compute the feed (back and forward) part of motor command.
#if SS == 0  // PID Controller
  float fCmd = Kp*error + Kd*dErrDt + Ki*errSum;
#else  // State-space controller
  float fCmd = (Kr*desired - (Kth*meas + Kw*dMeasDt + Ki*errSum));
#endif 
  float nomCmd = nomCmdIn + 0.5*propOffset; // Compute nominal

  // Write to H-bridge motor driver, FIX THIS FOR TWO-PROP CASE!
  float cmdClip = my6310.hbridgeWrite(nomCmd+fCmd, B2L); // B1L<->B2L 
  my6310.hbridgeWrite(nomCmd+fCmd, A2R); // A1R <-> A2R swaps direction

  // Writes to outer drivers (use BL to sub for AL, AR to sub for BR)
  // if there is a problem with one of the drivers.
  my6310.hbridgeWrite(nomCmd + fCmd, A2L); // A1L <-> A2L 
  my6310.hbridgeWrite(nomCmd + fCmd, B2R); // B1R <-> B2R 

  if(skipCounter > 0) skipCounter--;
  else {
    skipCounter = numSkipsIn;
    // Send data to monitor.
    float loopStatus[6];
    loopStatus[PLOT_TM] = meas;
    loopStatus[PLOT_TD] = desired; // headRoom
    loopStatus[PLOT_CMD] = cmdClip;
    loopStatus[PLOT_DDT] = dErrDt/10; 
    loopStatus[PLOT_SUM] = errSum/SumMax;
    my6310.sendStatus(5,loopStatus); 
  }
}




  


  
