/*
  Library for Teensy 4.1 6.310 board Created by teaching
  staff of 6.310, a control class at MIT.
*/

#include "lib6310.h"

extern ADC *_adc;
extern volatile uint16_t _bufMax;
extern volatile uint16_t _numSamples0;
extern volatile uint16_t _numSamples1;
extern volatile uint16_t _adcArray0[];
extern volatile uint16_t _adcArray1[];
extern volatile uint16_t _numS0;
extern volatile uint16_t _numS1;

void adc0_isr(void) {
  if(_numS0 < _numSamples0) {
    _adcArray0[_numS0] = (uint16_t) _adc->adc0->analogReadContinuous();
    _numS0++; 
  } 
  else _adc->adc0->analogReadContinuous();
};

void adc1_isr(void) {
  if(_numS1 < _numSamples1) {
    _adcArray1[_numS1] = (uint16_t) _adc->adc1->analogReadContinuous();
    _numS1++; 
  } 
  else _adc->adc1->analogReadContinuous();
};

lib6310::lib6310(int pwmRes, int adcRes, Stream *serialChan) {
// Set Dac and Adc resolution, and PWM frequency for library
// Set PWM frequency, Teensy 8bit PWM: 4.1-585937.5 
// Divide by 2,4,8 or 16 for 9,10,11, or 12 bits.
  int resDividerExp = max(0,pwmRes - 8);
  int resDivider = pow(2, resDividerExp);
  
  _PWMF = 585937.5/float(resDivider);
  _dacRes = pwmRes;
  _adcRes = adcRes;
  _serChan = serialChan;
  _dacMax = float(pow(2,_dacRes) - 1);
  _adcMax = float(pow(2,_adcRes) - 1);
  _firstTime = true;
  _inputTimer = 0;
  _inputPeriodFrac = 0.0;
  _saveDesired = 0;
  _inputString = "";
}

void lib6310::_parseConfigStr(String inStr) {
  // Count the number of plots
  int numPlots = 0;
  for(String subStr = inStr; subStr.length() > 5;) {
    int nextP = subStr.indexOf("P~");
    subStr = subStr.substring(nextP+2);
    int endLabel = subStr.indexOf('~');
    _plotLabels[numPlots] = subStr.substring(0,endLabel);
    subStr = subStr.substring(endLabel+1);
    _plotMins[numPlots] = subStr.toFloat();
    subStr = subStr.substring(subStr.indexOf('~')+1);
    _plotMaxs[numPlots] = subStr.toFloat();
    subStr = subStr.substring(subStr.indexOf('~')+1);
    subStr = subStr.substring(subStr.indexOf('~')+1);
    int numSubPlots = subStr.toInt();
    if(numPlots == 0) {
      _plotOffsets[numPlots] = 0.0;
    } else {
      _plotOffsets[numPlots] = _plotOffsets[numPlots-1]
	+ _plotMaxs[numPlots-1] + abs(_plotMins[numPlots]);
    }
    for(int j = 1; j < numSubPlots; j++) {
      int label_break = _plotLabels[numPlots].indexOf(" ");
      _plotLabels[numPlots+1] = _plotLabels[numPlots].substring(label_break+1);
      _plotLabels[numPlots] = _plotLabels[numPlots].substring(0,label_break);
      _plotMins[numPlots+1] = _plotMins[numPlots];
      _plotMaxs[numPlots+1] = _plotMaxs[numPlots];
      _plotOffsets[numPlots+1] = _plotOffsets[numPlots];
      numPlots += 1;
    }
    numPlots += 1;
  }
  _numPlots = numPlots;
}


// Enable both adc's,.
void lib6310::setupADC(int numAvg, boolean goFast)
{
  _adc->adc0->setAveraging(numAvg); 
  _adc->adc0->setResolution(_adcRes);
  if(goFast) {  // Go fast as possible
    _adc->adc0->setConversionSpeed(ADC_CONVERSION_SPEED::VERY_HIGH_SPEED); 
    _adc->adc0->setSamplingSpeed(ADC_SAMPLING_SPEED::VERY_HIGH_SPEED); 
  } else { // A little slower but less noisy
    _adc->adc0->setConversionSpeed(ADC_CONVERSION_SPEED::MED_SPEED); 
    _adc->adc0->setSamplingSpeed(ADC_SAMPLING_SPEED::MED_SPEED); 
  }
  _adc->adc0->stopContinuous();
  _adc->adc0->enableInterrupts(adc0_isr);

  
  _adc->adc1->setAveraging(numAvg); 
  _adc->adc1->setResolution(_adcRes);
  if(goFast) {  // Go fast as possible
    _adc->adc1->setConversionSpeed(ADC_CONVERSION_SPEED::VERY_HIGH_SPEED); 
    _adc->adc1->setSamplingSpeed(ADC_SAMPLING_SPEED::VERY_HIGH_SPEED); 
  } else { // A little slower but less noisy
    _adc->adc1->setConversionSpeed(ADC_CONVERSION_SPEED::MED_SPEED); 
    _adc->adc1->setSamplingSpeed(ADC_SAMPLING_SPEED::MED_SPEED); 
  }
  _adc->adc1->stopContinuous();
  _adc->adc1->enableInterrupts(adc1_isr);
}


// In: ADC pin, num samples in a burst.
// Out: Avg val of burst, individual vals in _adcSampleArray.
float lib6310::_adcReadIn(const uint8_t pin, const uint16_t numSamples) {

  // Initialize for average.
  float sumSample = 0.0;
  float avg = 0.0;

  if((pin != A12) && (pin != A13)) {
    // Wait until the samples are collected
    _numSamples0 = numSamples;  // Place where interrupt can see
    _adc->adc0->wait_for_cal();
    _numS0 = 0;
    _adc->adc0->startContinuous(pin);
    while(_numS0 < _numSamples0) {}  // Wait to get numSamples.
    _adc->adc0->stopContinuous();
    // Average Samples
    for(int i = 0; i < _numSamples0; i++) sumSample += _adcArray0[i];
    avg = (sumSample/float(_numSamples0));
    
  } else {
    // Wait until the samples are collected
    _numSamples1 = numSamples;  // Place where interrupt can see
    _adc->adc1->wait_for_cal();
    _numS1 = 0;
    _adc->adc1->startContinuous(pin);
    while(_numS1 < _numSamples1) {}  // Wait to get numSamples.
    _adc->adc1->stopContinuous();
    // Average Samples
    for(int i = 0; i < _numSamples1; i++) sumSample += _adcArray1[i];
    avg = (int) (sumSample/float(_numSamples1));
  }
  return(avg);
}

// Scales 0->adcMax to 0.0->1.0.
float lib6310::adcGetAvg(const uint8_t pin) {
  float avgADC = _adcReadIn(pin,1)/float(_adcMax);
  return(avgADC);
}

// Scales 0->adcMax to 0.0->1.0.
float lib6310::adcGetAvg(const uint8_t pin, uint16_t &numSamples) {
  uint16_t numSamplesMax = min(_bufMax,numSamples);
  numSamples = numSamplesMax;
  float avgADC = _adcReadIn(pin,numSamplesMax)/float(_adcMax);
  return(avgADC);
}
  
// Gets numSamples, scales 0->adcMax to 0.0->1.0.
float lib6310::adcGetBuf(const uint8_t pin, uint16_t &numSamples, float sampBuf[]) {
  uint16_t numSamplesMax = min(_bufMax,numSamples);
  numSamples = numSamplesMax;
  float avgADC = _adcReadIn(pin,numSamplesMax)/float(_adcMax);
  for(int i = 0; i < numSamplesMax; i++) sampBuf[i] = (_adcArray0[i])/_adcMax;
  return(avgADC);
}

// Setup the 6310 library.
void lib6310::setup(String config_message) {

  // Initialize Serial Channel
  _serChan->println(" ");
  while(!_serChan) {}
  _serChan->println(" ");
  _serChan->setTimeout(2);
  
  _inputString = "";
  _firstTime = true;
  _config_message = "\fB~" + config_message + "\r";

  // Set up analog inputs by disabling digital
  pinMode(IN1, INPUT_DISABLE);
  pinMode(IN2, INPUT_DISABLE);
  pinMode(IN3, INPUT_DISABLE);
  pinMode(IN4, INPUT_DISABLE);
  pinMode(IN5, INPUT_DISABLE);
  pinMode(IN6, INPUT_DISABLE);
  pinMode(IN7, INPUT_DISABLE);
  pinMode(IN8, INPUT_DISABLE);
  pinMode(IsenseAR, INPUT_DISABLE);
  pinMode(IsenseBR, INPUT_DISABLE);
  pinMode(IsenseAL, INPUT_DISABLE);
  pinMode(IsenseBL, INPUT_DISABLE);


  // Set up outputs.
  pinMode(loopMonitor, OUTPUT);
  analogWriteResolution(_dacRes);
  pinMode(A1R,OUTPUT); 
  pinMode(A2R,OUTPUT); 
  pinMode(B1R,OUTPUT); 
  pinMode(B2R,OUTPUT); 
  pinMode(A1L,OUTPUT); 
  pinMode(A2L,OUTPUT); 
  pinMode(B1L,OUTPUT); 
  pinMode(B2L,OUTPUT);
  pinMode(hEnablePWM, OUTPUT);
  pinMode(O1,OUTPUT); 
  pinMode(O2,OUTPUT); 
  pinMode(O3,OUTPUT); 
  pinMode(O4,OUTPUT);
  digitalWrite(hEnablePWM, LOW);
  analogWriteResolution(_dacRes); 
  analogWriteFrequency(A1R,_PWMF); 
  analogWriteFrequency(A2R,_PWMF); 
  analogWriteFrequency(B1R,_PWMF); 
  analogWriteFrequency(B2R,_PWMF); 
  analogWriteFrequency(A1L,_PWMF); 
  analogWriteFrequency(A2L,_PWMF); 
  analogWriteFrequency(B1L,_PWMF); 
  analogWriteFrequency(B2L,_PWMF);
  analogWriteFrequency(O1,_PWMF);
  analogWriteFrequency(O2,_PWMF);
  analogWriteFrequency(O3,_PWMF);
  analogWriteFrequency(O4,_PWMF);
  
  // Turn on and off loop monitor,diagnostic
  digitalWrite(loopMonitor,HIGH);
  delay(10);
  digitalWrite(loopMonitor,LOW);
  delay(10);
  digitalWrite(loopMonitor,HIGH);

}

// Turn on the 6310 PWM drivers
void lib6310::startPWM(float fraction) {
  pwmWrite(hEnablePWM,0.0);
  for(int i = 0; i < 3; i++) {
    analogWrite(A1R,_dacMax); 
    analogWrite(A2R,_dacMax);
    analogWrite(B1R,_dacMax); 
    analogWrite(B2R,_dacMax);
    analogWrite(A1L,_dacMax); 
    analogWrite(A2L,_dacMax);
    analogWrite(B1L,_dacMax); 
    analogWrite(B2L,_dacMax);
    if(i==0) pwmWrite(hEnablePWM,fraction);
  }
}


void lib6310::_processRcvString(String inStr, float rcvBuf[]) {
  //Serial.println("got here");
  //Serial.println(inStr);
  char St = inStr.charAt(0);
  inStr.remove(0,1);
  inStr.remove(0,1);

  float val = inStr.toFloat();
  switch (St) {
    case '0':
      rcvBuf[0] = val;
      //Serial.println(val);
      break;
    case '1':
      rcvBuf[1] = val;
      break;
    case '2': 
      rcvBuf[2] = val;
      break;
    case '3':
      rcvBuf[3] = val;      
      break;  
    case '4':
      rcvBuf[4] = val;      
      break;  
    case '5':
      rcvBuf[5] = val;      
      break;
    case '6':
      rcvBuf[6] = val;      
      break;
    case '7':
      rcvBuf[7] = val;      
      break;
    case '8':
      rcvBuf[8] = val;      
      break;
    case '9':
      rcvBuf[9] = val;      
      break;
    case '~':
      _firstTime = true;
      break;
    default:
    break;  
  }
}

// Load the serial output buffer.
void lib6310::sendStatus(int numStats, float stats[])
{
  if(_serChan == &Serial) { // Using the Arduino plotter
    _serChan->print(_plotLabels[0] + ":");
    _serChan->print(1000.0*(_plotOffsets[0]+stats[0]));
    for(int i=1; i < _numPlots; i++) {
      _serChan->print(",");
      _serChan->print(_plotLabels[i] + ":");
      _serChan->print(1000.0*(_plotOffsets[i]+stats[i]));
    }
    _serChan->println();
  }
  else {  // Not the Teensyduino plotter
    int n = 0;
    // Start Byte.
    _sendBuf[n++] = '\f';
    _sendBuf[n++] = 'R';

    // status data
    int dataBytes = numStats*sizeof(stats[0]);
    memcpy(&(_sendBuf[n]),&(stats[0]),dataBytes);
    n += dataBytes;

    // CR to end status buffer.
    _sendBuf[n++] = '\n';
  
    // write to the serial line.
    _serChan->write(_sendBuf,n);
  }
}

// Function to access the first time flag
boolean lib6310::isFirst()
{
  return(_firstTime);
}

// Function to access the first time flag
void lib6310::setFirst(boolean val)
{
  _firstTime = val;
}

float lib6310::getSlider(int whichVal)
{
  return(_rcvBuf[whichVal]);
}

float lib6310::getDacMax()
{
  return(_dacMax);
}

float lib6310::getAdcMax()
{
  return(_adcMax);
}

float lib6310::getInputPeriodFrac()
{
  return(_inputPeriodFrac);
}

// Initializes the loop if this is the first time, or if reconnect sent
// from GUI.  Otherwise, just look for serial event.
void lib6310::startloop() {
  if (_firstTime) { // If first time, send config string
    // Send the configuration info.
    if(_serChan == &Serial) { // Using the Arduino plotter
      _parseConfigStr(_config_message);
      _firstTime = false;
    } else {
      _inputString = "";
      _firstTime = false;
      // Flush Transmitter
      _serChan->flush();
      // Clear out rcvr.
      while(_serChan->available()) {
        _serChan->read(); 
      }
    }
    for(float timeOut = 0; !_serChan->available() && timeOut<1e6; timeOut++);
    while(_serChan->available() > 0) _serChan->read(); // Clr Rcv
    
  } 
  else { // Check for data from GUI
    while (_serChan->available()) {
      char inChar = (char)_serChan->read();
      _inputString += inChar;
      //Serial.println(inChar);
      if (inChar == '\n') { // newline terminates line.
        _processRcvString(_inputString,_rcvBuf);
        _inputString = "";
        break;
      } else if(_inputString.length() > 50) { // Comm failure.
        _inputString = "";
        break;
      }
    }

    // Flip the monitor pin every loop
    _monitorSwitch = !_monitorSwitch;
    digitalWrite(loopMonitor, _monitorSwitch);
  }
}

// Computes a new desired input value
float lib6310::updateDesired(float Amp, float Freq, float smooth) {
  float desired = Amp;

  // Record time since last called
  float delT = float(_inputTimer)*1.0e-6;
  _inputTimer = 0;

  // Determine fraction of period
  _inputPeriodFrac += delT*Freq;
  if(_inputPeriodFrac > 1.0) _inputPeriodFrac -= 1.0;
  desired *= sin(2*PI*_inputPeriodFrac);
  
  if(Amp > 0) { // If Amp > 0, turn sine into step with smoothing
    desired = (desired > 0) ? Amp : -Amp;
    desired = (1-smooth)*desired + smooth*_saveDesired;
    _saveDesired = desired;
  }
  return(desired);
}

// Write motor commands to hbridge.
// normalized (0->1) to sign-flipped hbridge, clipped (0, dMax).
float lib6310::hbridgeWrite(float normV, int pin) {
  float normVclip = min(max(normV,0.0),1.0);
  normVclip = 1.0 - pwmWrite(pin, 1.0-normVclip);
  return(normVclip);
}

#define minPWMfrac 0.01
float lib6310::pwmWrite(int pin, float normV) {
  // normalized (0->1) //
  float normVclip = min(max(normV,minPWMfrac),1.0);
  int unclip = int(normVclip*float(_dacMax));
  analogWrite(pin, unclip);
  return(normVclip);
}

// normV from -1->1, for maximum negative voltage to maximum positive.
// On pinP: normV from -oLap->1-oLap maps to dMax->0, 
// On pinN: normV from -1+oLap->oLap maps to dMax->0
#define oLapF 0.00  //overlap forward and backward by 5%
float lib6310::hbridgeBipolar(float normV, int pinP, int pinN) {
  float normVp = hbridgeWrite(normV + oLapF, pinP);
  float normVm = hbridgeWrite(oLapF - normV, pinN);
  float retval = normVp;
  if (normVm > normVp) retval = -normVm;
  return(retval);
}

// read voltage on pin, set to zero if close to zero.
float lib6310::analog2param(int pin) {
  float retval = adcGetAvg(pin) - zeroParamThresh;
  retval = scaleParam * max(retval, 0.0);
  return(retval);
}

// return val/scale bnded to be inbetween -Thresh and Thresh
float lib6310::scaleAndBound(float val, float Thresh, float Scale)
{
  if(Scale != 0.0) Thresh *= Scale;
  else Scale = 1.0;
  if(val > Thresh) val = Thresh;
  if(val < -Thresh) val = -Thresh;
  return(val/Scale);
}

backEMF::backEMF(int pVa, int pVb, int pC, float cS, float vS, float Rm)
{
  _pVa = pVa;
  _pVb = pVb;
  _pC = pC;
  _cScale = cS;
  _vScale = vS;
  _Rm = Rm;
}

float backEMF::bEMF(int nAvg, lib6310 lib)
{
  float bemfSum = 0.0;
  for(int i = 0; i < nAvg; i++) {
    float Vm = _vScale*(lib.adcGetAvg(_pVa) - lib.adcGetAvg(_pVb));
    float crnt = _cScale*lib.adcGetAvg(_pC);
    bemfSum += abs(Vm) - (_Rm*crnt); //backEmf = motorV - Rm * motor_current;
  }

  // normBemf = bEmf/5v 
  float normBemf = bemfSum/(nAvg*_vScale);
  return(normBemf);
}
