// ****************** MUST INCLUDE LINES BELOW*****************************
#include <lib6310.h>
// Init library and set DAC resolution, ADC resolution, serial channel
#define MYSERIAL Serial
lib6310 my6310(12,12,&Serial);
// Set up the analog-to-digital library, with a buffersize
#define sampleBufN 500
LIB6310_ADC(sampleBufN+100);  // ADC sample buffer of 100+sampleBufN
// ****************** MUST INCLUDE LINES ABOVE*******1**********************

// Constants for Users to Modify 
// ************************************************************
#define AngleOffset 0.0 // Adjust if measured angle is not zero when horizontal. 
#define Psmooth 0.0 // Toggle smoothing 0: no smooth, 0.9->0.999: less->more smooth
#define DeltaT_Microsecs 1000 // Update period in microseconds
#define nomCmd 0.4 // Motor command that nominally hold arm level.
#define MbackMax 250  // Maximum value of mBack
#define NUMMOTORS 2 // Number of propeller motors
#define MODE 0 // 0: User Theta_d 1,2,3:Sweeper Theta_d  1:Waveplot 2:FreqRes 3:Matlab Print

// Frequency Sweep Parameters
#define firstFreq 0.5
#define lastFreq 5
#define MAX_FREQ_PTS 40

// Indexes for values being sent from plotter send window.
// e.g. to set KP to 0.3, type "3 0.3" in the send window and click send, 
// e.g. to set the input period to two seconds, type "5 0.5" and click send.
#define EXTRABACK 0  // dx/dt is approx (x[n]-x[n-m])/(DeltaT*m), m = 1+extraBack
#define KD 1         // Derivative Gain
#define KI 2         // Integral Gain
#define KP 3         // Proportional Gain
#define SKIPS 4      // Number of sample points skipped between plotted points
#define FREQ 5       // Input frequency, only used in MODE 0
#define AMP 6        // Input peak amplitude, Amp>0:step inputs, Amp<0:sine inputs
#define PAUSE 7      // Pause sweep in Mode 3, with "7 1", unpause with "7 0"

// offsets for parameters sendable from plotter window.
#define offsetExtraBack 5
#define offsetKd 0.0
#define offsetKi 0.0
#define offsetKp 0.0
#define offsetNumSkips 19
#define offsetFreq 0.0
#define offsetAmp 0.0

// ************************************************************************
// End of Constants and code for Users to Modify 
// SEE END OF FILE FOR USER-ALTERABLE CONTROL LOOP!!

// *******SETTING UP THE MONITOR DISPLAY (SLIDERS AND PLOTS). *************
// Sliders or settings from plotter window.
String sliders = "S~ExtraBack~0~20~S~Kd~0~5~S~Ki~0~100~S~Kp~0~10~S~Skips~0~50~S~Freq~-3~3~S~Amp~-3~3~S~Pause~-1~1~";

#if (MODE == 2)
  String monPlots = "P~Freq CmdRe CmdIm OutRe OutIm~-10~10~500~5~";
#else
  String monPlots = "P~Theta_m Theta_d Cmd~-1~1~500~3~P~DEdt/10 ErrSum~-1~1~500~2~";
#endif

#define PLOT_TM 0
#define PLOT_TD 1
#define PLOT_CMD 2
#define PLOT_DDT 3
#define PLOT_SUM 4

#define PLOT_F 0
#define PLOT_CMD_S 1
#define PLOT_CMD_C 2
#define PLOT_OUT_S 3
#define PLOT_OUT_C 4
#define PLOT_IN_S 5
#define PLOT_IN_C 6


// Configuration string with surrounding limits.
String config_message = sliders + monPlots;

// Control Loop Input Functions
// ************************************************************************

// Measures Angle in Radians
float getTheta() {
  uint16_t numSamp = 9;
  float theta = 0.0;
  for(int i = numSamp; i > 0; i--) {
    theta += 2*PI*(my6310.adcGetAvg(IN4) - 0.5);
  }
  theta = theta/float(numSamp) - AngleOffset;
  return(theta);
}

// Approximates d(x)/dt as (x[n] - x[n-m])/(m*deltaT)
float dXdt(float x, int m, float delT)
{
  static float xBack[MbackMax+1];  // Stores past errors for deriv approx

  m = max(m,1); // Make sure m is at least 1.

  // save past errors.
  for (int i = MbackMax; i > 0; i--) xBack[i] = xBack[i-1];
  xBack[0] = x;

  // Calculate approximation to d(x)/dt
  float dxDtVal = (x- xBack[m])/(float(m)*delT); 

  return(dxDtVal);
}

// Approximately integrates x with bounds.
float errSum = 0.0;  // Stores accumulated errors.
float lastKi = 0.0;  // Used to zero sum when Ki changes.
// Computes the deltaT weight sum of errors.
float integX(float x, float Ki, float delT)
{
  static float xSum = 0.0;
  static float lastKi = 0.0;

  if(Ki > 0) {
    if(lastKi != Ki) xSum = 0.0;
    float SumMax = 100/(0.1 + Ki);
    xSum += x * delT; 
    xSum = max(min(SumMax,xSum), -SumMax);
  }
  lastKi = Ki;
  return(xSum);
}

// Update input when using sweeper
#define deg2rad 0.0175
#define TSSS 4.0 // Max num seconds to steady-state
boolean Pause = false;

float updateSweeper(float cmd, float theta, float theta_d, float delT) {
  static int numDts = 0;
  static int periodDts = 0;
  static int sssDTs = 0;
  static int freqIndex = 0;
  static int plotIndex = 0;
  static float Freq = 0.0;
  static float sinv = 0.0;
  static float cosv = 0.0;
  static float sc = 0.0; 
  static float cc = 0.0;
  static float sa = 0.0;
  static float ca = 0.0;
  static float sd = 0.0;
  static float cd = 0.0;
  static float Fmultiplier = exp(log(lastFreq/firstFreq)/float(MAX_FREQ_PTS));

  // Update projections uses sinv and cosv from last call.
  sc += cmd*sinv; cc += cmd*cosv; 
  sa += theta*sinv; ca += theta*cosv;
  sd += theta_d*sinv; cd += theta_d*cosv;
  
  if( ((numDts % periodDts) == 0) || (numDts >= sssDTs) ) { //period Bound
    if(numDts >= sssDTs) {  // Ready to save steady-state
      // Save mag and phase if requested.
      if (MODE == 2) {
        float loopStatus[7];
        // Divide by 1000 to undo library scaling
        float scaleSum  = sd*1000;
        loopStatus[PLOT_F] = Freq/1000; 
        loopStatus[PLOT_CMD_S] = sc/scaleSum;
        loopStatus[PLOT_CMD_C] = cc/scaleSum;
        loopStatus[PLOT_OUT_S] = sa/scaleSum;
        loopStatus[PLOT_OUT_C] = ca/scaleSum;
        loopStatus[PLOT_IN_S] = sd/scaleSum;
        loopStatus[PLOT_IN_C] = cd/scaleSum;
        my6310.sendStatus(5,loopStatus);
      }

      if (MODE == 3 && !Pause) {
        Serial.print(Freq);
        Serial.print(",");
        Serial.print(sc/sd,6);
        Serial.print(",");
        Serial.print(cc/sd,6);
        Serial.print(",");
        Serial.print(sa/sd,6);
        Serial.print(",");
        Serial.print(ca/sd,6);
        Serial.print("; ");
        if(plotIndex == 4) {
          plotIndex = 0;
          Serial.println(" ");
        } else plotIndex++;
      }

      // Update frequency
      if((freqIndex == 0) || (freqIndex > MAX_FREQ_PTS)) { 
        Freq = firstFreq;
        freqIndex = 0; 
      } 
      else Freq *= Fmultiplier;
      freqIndex += 1;

      // Tweak frequency to be exactly N deltaT's
      periodDts = 2*int(0.5/(delT*Freq));
      Freq = float(1.0)/(delT*float(periodDts)); 

      // Wait for SS for 3 periods or TSSS, whichever is larger.
      int oscPeriods = 1+int(TSSS/(delT*float(periodDts)));
      sssDTs = max(oscPeriods,3) * periodDts; // Wait at least 3 periods for SSS
      numDts = 0;
    }
    // Reset projection parameters at end of each period.
    sc = 0; cc = 0; sa = 0; ca = 0; sd = 0; cd = 0;
  } // End of period boundary case.

  numDts += 1;
  float arg = 2*PI*float(numDts)/float(periodDts);
  sinv = sin(arg);
  cosv = cos(arg);
  
  return(sinv);
}

// Get Theta in radians
elapsedMicros loopTimer = 0;  // Loop runs at least 10xsecond
float deltaT = 0.0;
float headRoom;
void setup() {
  my6310.setup(config_message);
  my6310.startPWM(0.98); // start parallel drive pwm, A+B in sync.
  my6310.setupADC(8,false);
  loopTimer = 0;
  deltaT = float(DeltaT_Microsecs)*1.0e-6;
}

//********************** Main USER ALTERABLE Control Loop **********************
//**************************************************************************
int skipCounter = 0;  // Counts plot-skipped samples.
float theta_dNext = 0.0;


void loop() {  
  // Initialize and check for usb inputs.
  my6310.startloop(); 

  // Get the nominal propeller command from the analog input.
  float nomCmdL = 0.1*(my6310.analog2param(IN8) - 0.5);

  // Get send window parameters.
  int extraBack = offsetExtraBack + my6310.getSlider(EXTRABACK); 
  float Kp = offsetKp + my6310.getSlider(KP); 
  float Kd = offsetKd + my6310.getSlider(KD); 
  float Ki = offsetKi + my6310.getSlider(KI);
  int numSkips = offsetNumSkips + my6310.getSlider(SKIPS);
  float freq =  offsetFreq + my6310.getSlider(FREQ);
  float amp = offsetAmp + my6310.getSlider(AMP);
  Pause = (my6310.getSlider(PAUSE) > 0);


  // Headroom is faction of period left until next sensor read.
  //float headRoom = float(DeltaT_Microsecs - int(loopTimer));

  // Wait exactly one deltaT since last sensor read.
  while (int(loopTimer) < DeltaT_Microsecs) {};
  loopTimer = 0;

  // Get the desired and measured arm angle.
  float theta_d = theta_dNext; 
  if(MODE == 0) theta_d = my6310.updateDesired(amp,freq,Psmooth);
  float theta_a = getTheta();

  // Compute the error and its approximate derivative and integral.
  float Err = (theta_d - theta_a);
  float dErrDt = dXdt(Err, 1+extraBack, deltaT);
  float intErr = integX(Err, Ki, deltaT);

  // Compute the offset-from-nominal motor cmd 
  float dCmd = Kp*Err + Kd*dErrDt + Ki*intErr;

  // Bound command and write to H-bridge 
  dCmd = my6310.scaleAndBound(dCmd, 1.0-nomCmd, 1.0); // Ensure symmetric bound.

  float cmdClip = my6310.hbridgeBipolar(nomCmd + dCmd + nomCmdL, B2L, B1L); // B1L <-> B2L swaps direction
#if (NUMMOTORS == 2) 
  my6310.hbridgeBipolar((nomCmd - dCmd), A2R, A1R); // A1R <-> A2R swaps direction
#endif
  if(MODE > 0) theta_dNext = amp*updateSweeper(dCmd, theta_a, theta_d, deltaT);

  // Display the results 
  if(skipCounter > 0) skipCounter--;
  else {  // Send data to monitor.
    skipCounter = numSkips;
    if (MODE < 2) {
      float loopStatus[5];
      loopStatus[PLOT_TM] = theta_a;
      loopStatus[PLOT_TD] =  theta_d; //headRoom;
      loopStatus[PLOT_CMD] = cmdClip;
      loopStatus[PLOT_DDT] = dErrDt/10;
      loopStatus[PLOT_SUM] = intErr/(0.1 + Ki);
      my6310.sendStatus(5,loopStatus);
    }
  }

}




  


  
