// ****************** LINES BELOW: INCLUDE LIBRARIES AND ADC INITIALIZATION. **************
#include <lib6310.h>
#define MYSERIAL Serial
lib6310 my6310(11,12,&Serial);  // 11-bit DAC, 12-bit ADC.
#define sampleBufN 500 // analog-to-digital library buffersize
LIB6310_ADC(sampleBufN+100);  // ADC sample buffer of 100+sampleBufN
// ****************** LINES ABOVE: INCLUDE LIBRARIES AND ADC INITIALIZATION. **=*************


// Constants Users are likely to Modify (or know about)
#define SensorScale 0.85   // Scale distance sensor so that SensorScale*(sensVal(min) - sensVal(min+1.5cm)) = -1500 (microns).
#define SensorOffset -2.5 // Offset the scaled sensor value so SensorScale*(sensVal(min+1.5cm)) - SenseOffset = 0 
#define crntSign 1.0      //  +1.0,-1.0 Sign of the coil current sensor
#define crntOffset 0.43 // Remove nominal current value.
#define Psmooth 0.0     //Smooths input steps: 0.0: no smoothing  0.999: very smooth

//*************************************************************************************
//******************* NOMINAL VALUES FOR SEND WINDOW SETTABLE PARAMETERS **************
// Below are nominal values for parameters that can be modified using
// the Arduino's serial plotter send window.  NOTE: A signed value sent from 
// the serial plotter is ADDED its associated offset below.  
// MULTIPLE SENDS DO NOT ACCUMULATE!!!

// COPY AND PASTE NOMINAL GAIN VALUES FROM MATLAB HERE!

// Gains not defined by matlab copy+paste are zeroed below.
#ifndef offsetKicoil
#define offsetKicoil 0.0  
#endif
#ifndef offsetKy
#define offsetKy 0.0  
#endif
#ifndef offsetKv
#define offsetKv 0.0  
#endif
#ifndef offsetKr
#define offsetKr 0.0 
#endif
#ifndef offsetKint
#define offsetKint 0.0  
#endif

// Offsets for plotting and inputs
#define offsetExtraBack 3
#define offsetNumSkips 19

// Offsets used for Input frequency and Amplitude
#define offsetFreq 0.3    
#define offsetAmp 0.0   

//******************* END OF NOMINAL VALUES SEND WINDOW SETTABLE PARAMETERS
//*************************************************************************************

// Constants only advanced users should modify.
#define offsetNomCmd 0.0  // nom command = nomCmdOffset+potscale*(potval)
#define potScale 20.0     // Increase->wider range, decrease->finer correction
#define controllerUpdatePeriod 200  // Sampling 10,000 times/second!!!
#define MbackMax 250 // dMeas/dt est by (Meas[n]-Meas[n-Mback] )/(DeltaT*Mback)
#define PLOTSEL 0 // Switch to 1 ONLY TO CHECK CONTROL LOOP HEADROOM!

// ********************* End of Constants and code for Users to Modify 
// SEE END OF FILE FOR USER-ALTERABLE CONTROL LOOP!!

// Indexes for values set using the plotter send window.
// e.g. to set KY to 0.3, type "3 0.3" in the send window and click send, 
// e.g. to set the input period to two seconds, type "5 0.5" and click send.
#define EXTRABACK 0  // velocity is approx (y[n]-y[n-m])/(DeltaT*m), m = 1+extraBack
#define KV 1         // Velocity Gain
#define KINT 2       // Integral Gain
#define KY 3         // PositionGain
#define SKIPS 4      // Number of sample points skipped between plotted points
#define FREQ 5       // Input frequency, only used in MODE 0
#define AMP 6        // Input peak amplitude, Amp>0:step inputs, Amp<0:sine inputs
#define KICOIL 7     // Coil current gain
#define KR 8         // Scaling of y_desired, Should be set carefully!

// *******SETTING UP THE MONITOR DISPLAY (SLIDERS AND PLOTS). *************
// Sliders
String sliders = "S~ExtraB~0~10~S~Kv~0~2~S~Kint~0~200~S~Ky~0~20~S~Skips~0~500~S~Freq~0.1~40~S~Amp~-1~1~S~Kc~0~10~S~Kr~0~20~";

#define NUMPLOTS 6
#define PLOT_M 0
#define PLOT_D 1
#define PLOT_CRNT 2
#define PLOT_VEL 3
#define PLOT_CMD 4 

String monPlotsIntro = "P~dY_m dY_d Crnt~-1~1~500~3~P~V/10 Cmd~-1~1~500~2~";

#if PLOTSEL == 0
  String monPlotsFinal = "P~IntErr~-1~1~500~2~";  
  #define PLOT_INTERR 5
  #define PLOT_HR 6
#else
  String monPlotsFinal =  "P~HR~-1~1~500~2~";  
  #define PLOT_HR 5
  #define PLOT_INTERR 6
#endif

// Configuration string with surrounding limits.
String config_message = sliders + monPlotsIntro + monPlotsFinal;

// Function turns off coil if command is a max value for too long.
#define TURNOFFTIMEOUT 1000000
#define cmdThresh 1.0
float reduceHeat(float cmd) {
  static uint32_t turnOffCnt = 0;  // static means variable persists.
  if((cmd > cmdThresh) || (cmd < -cmdThresh)) turnOffCnt++;
  else turnOffCnt = 0;
  if(turnOffCnt > TURNOFFTIMEOUT) cmd = 0.0;
  return(cmd);
}

// Control loop setup
// ************************************************************************
elapsedMicros loopTimer = 0;  // Loop runs at least 10xsecond
float deltaT = 0.0;
int sLpCntr = 0;  // Loop counter for sinusoid.
int dispLoopCounter = 0;

void setup() {
  my6310.setup(config_message);
  my6310.startPWM(1.0); // full scale
  my6310.pwmWrite(O3, 1.0);
  my6310.pwmWrite(O4, 0.0);
  my6310.setupADC(8,false);
  loopTimer = 0;
  dispLoopCounter = 0;
  deltaT = float(controllerUpdatePeriod)*1.0e-6;
}

//********************** Main USER ALTERABLE Control Loop **********************
//**************************************************************************
float pastDeltaY[MbackMax+1];   // Past Distances, used to approx velocity.
float cmdCL = 0.0;              // Clipped command.
float intErr = 0.0;             // Integrated error.
float lastKint = 0.0;           
float Cnom  = 0;

void loop() {  

  my6310.startloop(); // Initialize and check for usb inputs.

  // Get parameters
  int extraBack = offsetExtraBack + my6310.getSlider(EXTRABACK); 
  int Mback = 1 + max(extraBack,0);
  float Kv = offsetKv + my6310.getSlider(KV);
  float Kint = offsetKint + my6310.getSlider(KINT);
  float Ky = offsetKy + my6310.getSlider(KY);
  int numSkipsIn = offsetNumSkips + my6310.getSlider(SKIPS);
  float freqIn =  offsetFreq + my6310.getSlider(FREQ);
  float ampIn = offsetAmp + my6310.getSlider(AMP);
  float Kicoil = offsetKicoil + my6310.getSlider(KICOIL);
  float Kr = offsetKr + my6310.getSlider(KR);

  float headRoom = float(controllerUpdatePeriod - int(loopTimer)); // Headroom = excess time between loops
  while (int(loopTimer) < controllerUpdatePeriod) {}; // Start loop on time.
  loopTimer = 0;

 // Measure field sensor, estimate coil current.
  float crnt =  crntSign*10.0*(my6310.adcGetAvg(IN1) - crntOffset);

  // Measure IR distance sensor, 
  float sensVal = my6310.adcGetAvg(IN5); 

  // Compute estimate of umbrella distance
  sensVal = min(max(sensVal,0.05),1.0);
  //float deltaY = SensorOffset - SensorScale*sensVal;
  float deltaY = SensorOffset + SensorScale/sensVal;

  // Compute desired umbrella distance.
  float deltaY_d = my6310.updateDesired(ampIn,freqIn,Psmooth);

  // Save past values
  for (int i = Mback+1; i > 0; i--) pastDeltaY[i] = pastDeltaY[i-1];  
  pastDeltaY[0] = deltaY;
  
  // Compute approximate velocity 
  float mDeltaT = float(Mback)*deltaT;
  float delDeltaY = (deltaY - pastDeltaY[Mback]); 
  #define BV 200.0
  float vel = my6310.scaleAndBound(delDeltaY, BV, mDeltaT);
 
  // Compute Error Integral 
  if(Kint == 0 || Kint != lastKint) intErr = 0;  // Reset error if Kint=0 or changed.
  else {
    float intMax = 200.0/(abs(Kint)+1); 
    float error = (deltaY - deltaY_d); 
    intErr = my6310.scaleAndBound(intErr+deltaT*error, intMax, 1.0);
  }
  lastKint = Kint;

  // Feedback 
  float deltaCmd = Kr*deltaY_d - (Ky*deltaY+Kv*vel+Kicoil*crnt+Kint*intErr);

  // Add in nominal command.
  float coilCmd = Cnom + deltaCmd;

  // Turn off coil cmd if on too long.
  coilCmd = reduceHeat(coilCmd);

  // Write coilCmd to three coils, save clipped version.
  cmdCL = my6310.hbridgeBipolar(coilCmd, B1L,B2L); // B2L <-> B1L swaps direction
  my6310.hbridgeBipolar(coilCmd, A2L,A1L);   // A1L<-> A2L swaps direction
  my6310.hbridgeBipolar(coilCmd, A2R,A1R); // A1R <-> A2R swaps direction
  
  // Status and nominal updates every plotincrement cycle.
  if(dispLoopCounter <  numSkipsIn) dispLoopCounter++;
  else {
    dispLoopCounter = 0;
    // Potentiometer read 
    Cnom = offsetNomCmd + potScale*(my6310.analog2param(IN8) - 0.5);

    // Send data to monitor.
    float loopStatus[7];
    loopStatus[PLOT_M] = deltaY;
    loopStatus[PLOT_D] = deltaY_d;
    loopStatus[PLOT_VEL] = vel/10.0;
    loopStatus[PLOT_CRNT] = crnt;
    loopStatus[PLOT_CMD] = cmdCL;
    loopStatus[PLOT_INTERR] = intErr;
    loopStatus[PLOT_HR] = (headRoom/1000);
    my6310.sendStatus(NUMPLOTS,loopStatus); 
  }
}




  


  
