// ****************** MUST INCLUDE LINES BELOW*****************************
#include <lib6310.h>
#define MYSERIAL Serial
lib6310 my6310(11,12,&Serial);  // 11-bit DAC, 12-bit ADC.
#define sampleBufN 500 // analog-to-digital library buffersize
LIB6310_ADC(sampleBufN+100);  // ADC sample buffer of 100+sampleBufN
// ****************** MUST INCLUDE LINES ABOVE*******1**********************

// Constants for Users to Modify (or know about)
#define MODE 0 // MODE 0: Test Coil Dynamics 1:Maglev Control
#define SensorScale 1.0 // SensorScale*(sensVal(min)-sensVal(min+1.5cm))=1500
#define SensorOffset 0.0// SensorScale*(sensVal(min+1.5cm)) - SenseOffset = 0 
#define CurrentOffset 0.5 // Offset current sensor to zero when field is zero
#define controllerUpdatePeriod 100  // Sampling 10,000 times/second!!!
#define PLOTSEL 0 // 0:d,ddes,cmd 1:d,ddes,dErr/dt 2:d,V,Crnt 3:d,intErr,headrm
#define offsetNomCmd 0.0  // nominal command = nomCmdOffset+potscale(potval)
#define potScale 20.0

// Indexes for values being sent from plotter send window.
// e.g. to set KP to 0.3, type "3 0.3" in the send window and click send, 
// e.g. to set the input period to two seconds, type "5 0.5" and click send.
#define EXTRABACK 0 // estimate dx/dt = (x[n]-x[n-m])/(DeltaT*m), m=1+extraBack
#define KD 1        // Derivative Gain
#define KI 2        // Integral Gain
#define KP 3        // Proportional Gain
#define SKIPS 4     // Number of sample points skipped between plotted points
#define FREQ 5      // Input frequency, only used in MODE 0
#define AMP 6       // Input peak amp, Amp>0:step inputs, Amp<0:sine inputs
#define CTYPE 7     // Controller Type 0:PD, 1:Lead, 2:state-Space

// Initial offsets for parameters sendable from plotter window.
#define offsetExtraBack 3
#define offsetKd 0.0
#define offsetKi 0.0
#define offsetKp 0.0
#define offsetNumSkips MODE*99
#define offsetFreq 0.0
#define offsetAmp (MODE - 1.0)*0.9
#define offsetCtype 0

// State Space controller gains
#define Krss 0.0  // Input scaling, Be Sure to Set!!!!›
#define Kiss 0.0  // Integral
#define Kdss 0.0  // Distance
#define Kvss 0.0  // vel = dDist/dt
#define Kcss 0.0  // Crnt (prop to dVel/dt)

// Constants for advanced users to modify.
#define Psmooth 0.0 //0.0: no smoothing  0.999: very smooth
#define MbackMax 250 // dMeas/dt est by (Meas[n]-Meas[n-Mback] )/(DeltaT*Mback)
// ************************************************************************
// End of Constants and code for Users to Modify 
// SEE END OF FILE FOR USER-ALTERABLE CONTROL LOOP!!

// *******SETTING UP THE MONITOR DISPLAY (SLIDERS AND PLOTS). *************
// Sliders
String sliders = "S~ExtraB~0~50~S~Kd~0~5~S~Ki~0~100~S~Kp~0~10~S~Skips~0~50~S~Freq~0.1~20~S~Amp~-3~3~S~Ctype~0~3~";

// Plots: USE SPACES BETWEEN LABELS FOR MULTIPLOTS!!
#if MODE == 0
String monPlots = "P~Crnt Cmd~-1~1~500~2~"; 
#define PLOT_CRNT 0
#define PLOT_D 2
#define PLOT_CMD 1
#define PLOT_DDT 3
#define PLOT_VEL 3
#define PLOT_M 3
#define PLOT_INTERR 3 
#define PLOT_HR 3
#elif PLOTSEL == 0
String monPlots = "P~DY_m DY_d~-1~1~500~2~P~Cmd~-1~1~500~1~"; 
#define PLOT_M 0
#define PLOT_D 1
#define PLOT_CMD 2
#define PLOT_DDT 3
#define PLOT_VEL 3
#define PLOT_CRNT 3
#define PLOT_INTERR 3 
#define PLOT_HR 3
#elif PLOTSEL == 1
String monPlots = "P~10xDY_m 10xDY_d~-1~1~500~2~P~dErrDt/10~-1~1~500~1~";
#define PLOT_M 0
#define PLOT_D 1
#define PLOT_DDT 2
#define PLOT_VEL 3
#define PLOT_CRNT 3
#define PLOT_CMD 3
#define PLOT_INTERR 3 
#define PLOT_HR 3
#elif PLOTSEL == 2
String monPlots = "P~10xDY_m~-1~1~500~1~P~Vel~-1~1~500~1~P~Crnt~-1~1~500~1~";
#define PLOT_M 0
#define PLOT_VEL 1
#define PLOT_CRNT 2
#define PLOT_D 3
#define PLOT_DDT 3
#define PLOT_CMD 3
#define PLOT_INTERR 3 
#define PLOT_HR 3
#else
String monPlots = "P~10xDY_m IntErr HR~-1~1~500~3~";
#define PLOT_M 0
#define PLOT_INTERR 1
#define PLOT_HR 2
#define PLOT_D 3
#define PLOT_DDT 3
#define PLOT_VEL 3
#define PLOT_CRNT 3
#define PLOT_CMD 3
#endif

// Configuration string with surrounding limits.
String config_message = sliders + monPlots;

// Function turns off coil if command is a max value for too long.
#define TURNOFFTIMEOUT 1000000
#define cmdThresh 1.0
float reduceHeat(float cmd) {
  static uint32_t turnOffCnt = 0;  // static means variable persists.
  if((cmd > cmdThresh) || (cmd < -cmdThresh)) turnOffCnt++;
  else turnOffCnt = 0;
  if(turnOffCnt > TURNOFFTIMEOUT) cmd = 0.0;
  return(cmd);
}

// Control loop setup
// ************************************************************************
elapsedMicros loopTimer = 0;  // Loop runs at least 10xsecond
float mDeltaT = 0.0;
float deltaT = 0.0;
int sLpCntr = 0;  // Loop counter for sinusoid.
int dispLoopCounter = 0;

void setup() {
  my6310.setup(config_message);
  my6310.startPWM(1.0); // full scale
  my6310.pwmWrite(O3, 1.0);
  my6310.pwmWrite(O4, 0.0);
  my6310.setupADC(8,false);
  loopTimer = 0;
  dispLoopCounter = 0;
  deltaT = float(controllerUpdatePeriod)*1.0e-6;
}

//********************** Main USER ALTERABLE Control Loop **********************
//**************************************************************************
float pastError[MbackMax+1];  // Past errors, used to approximate DErr/dt)
float pastDeltaY[MbackMax+1];   // Past Distances, used to approx velocity.
float pastDeltaC[MbackMax+1];    // Past Command values.
float cmdCL = 0.0;          // Clipped command.
float intErr = 0.0;               // Integrated error.
float lastKi = 0.0;           
float Cnom  = 0;

void loop() {  

  my6310.startloop(); // Initialize and check for usb inputs.

  // Get parameters
  int extraBack = offsetExtraBack + my6310.getSlider(EXTRABACK); 
  int Mback = 1 + max(extraBack,0);
  float Kd = offsetKd + my6310.getSlider(KD);
  float Ki = offsetKi + my6310.getSlider(KI);
  float Kp = offsetKp + my6310.getSlider(KP);
  int numSkipsIn = offsetNumSkips + my6310.getSlider(SKIPS);
  float freqIn =  offsetFreq + my6310.getSlider(FREQ);
  float ampIn = offsetAmp + my6310.getSlider(AMP);
  int controllerTypeIn = offsetCtype + my6310.getSlider(CTYPE);

  float headRoom = float(controllerUpdatePeriod - int(loopTimer)); // Headroom = excess time between loops
  while (int(loopTimer) < controllerUpdatePeriod) {}; // Start loop on time.
  loopTimer = 0;

  // Get measured and desired distance to umbrella, compute error.
  float sensVal = my6310.adcGetAvg(IN5); 
  sensVal = min(max(sensVal,0.05),1.0);
  float deltaY = SensorOffset - SensorScale*sensVal;
  //float deltaY = SensorOffset + SensorScale/sensVal;
  float deltaY_d = my6310.updateDesired(ampIn,freqIn,Psmooth);
  float error = (deltaY_d - deltaY); 

  // Save past Errors.
  for (int i = Mback+1; i > 0; i--) { // Mback > 0, so Mback+1 >= 2.
    pastDeltaY[i] = pastDeltaY[i-1];
    pastError[i] = pastError[i-1];
    pastDeltaC[i] = pastDeltaC[i-1];
  }
  pastDeltaY[0] = deltaY;
  pastError[0] = error;

  // Compute approximate velocity and d(error)/dt
  mDeltaT = float(Mback)*deltaT;
  float vel = (deltaY - pastDeltaY[Mback]); 
  float dErrDt = (error - pastError[Mback]); 
  #define BV 200.0
  vel = my6310.scaleAndBound(vel, BV, mDeltaT);
  dErrDt = my6310.scaleAndBound(dErrDt, BV, mDeltaT);

  // Compute Integral Approximation
  float intMax = 200.0/(Kiss+1); 
  if(Ki != lastKi) { intErr = 0; lastKi=Ki; } // Zero integral if change in Ki.
  intErr = my6310.scaleAndBound(intErr+deltaT*error, intMax, 1.0);

  // Read magnetic field sensor, crnt is proportional to field.
  float crnt = 0.0;
  crnt =  10.0*(my6310.adcGetAvg(IN1) - CurrentOffset);
  
  // Feedback control using one of three controllers.
  float deltaC = 0;  // Dc for Delta c.
  if(controllerTypeIn == 0) { // 0: PD controller.
    deltaC = Kp*error + Kd*dErrDt + Ki*intErr; 
  }
  else if(controllerTypeIn == 2) { // 2: State-Space Controller.
    deltaC = Krss*deltaY_d - (Kdss*deltaY + Kvss*vel + Kcss*crnt - Kiss*intErr);
  }
  else if(controllerTypeIn == 1) { // 1: Lead controller
    deltaC = 0.0*pastDeltaC[1] + 0.0*error + 0.0*pastError[1];
  }
  pastDeltaC[0] = deltaC;
  float coilCmd = Cnom + deltaC;

  // Overwrite coil command if in test mode.
  if(MODE == 0) coilCmd = deltaY_d;
  
  // Avoid heating coils by reducing command if maxed out for too long.
  coilCmd = reduceHeat(coilCmd);

  // Write coilCmd to three coils, save clipped version.
  cmdCL = my6310.hbridgeBipolar(coilCmd, B1L,B2L); // B2L <-> B1L swaps dir
  my6310.hbridgeBipolar(coilCmd, A2L,A1L);   // A1L<-> A2L swaps dir
  my6310.hbridgeBipolar(coilCmd, A2R,A1R); // A1R <-> A2R swaps dir
  
  // Status and nominal updates every plotincrement cycle.
  if(dispLoopCounter <  numSkipsIn) dispLoopCounter++;
  else {
    dispLoopCounter = 0;

    // Potentiometer read 
    Cnom = offsetNomCmd + potScale*(my6310.analog2param(IN8) - 0.5);

    // Send data to be plotted. All data is multiplied by 1000
    // before being transmitted.
    float loopStatus[4];
    loopStatus[PLOT_M] = deltaY;
    loopStatus[PLOT_D] = deltaY_d;
    loopStatus[PLOT_DDT] = dErrDt/10.0;
    loopStatus[PLOT_VEL] = vel;
    loopStatus[PLOT_CRNT] = crnt;
    loopStatus[PLOT_CMD] = cmdCL;
    loopStatus[PLOT_INTERR] = intErr;
    loopStatus[PLOT_HR] = (headRoom/1000);
    my6310.sendStatus(2+MODE,loopStatus); 
  }
}




  


  
