// ****************** MUST INCLUDE LINES BELOW*****************************
#include <lib6310.h>
// Init library and set DAC resolution, ADC resolution (12 bits ONLY for Teensy 4.1), serial channel.
 #define MYSERIAL Serial
lib6310 my6310(11,12,&Serial);
// Set up the analog-to-digital library, with a buffersi    e
#define sampleBufN 500
LIB6310_ADC(sampleBufN+100);  // ADC sample buffer of 500
// ****************** MUST INCLUDE LINES ABOVE*******1**********************

// Constants for Users to Modify 
// ***********************************************************
#define MODE 0 // 0: oscilloscope mode, 1 = control mode
#define DisturbA 0.0 // Disturbance Amplitude (0.0->0.7).
#define nomOmega 15  // Nominal speed (Omega) in RPS
#define nomCommand 0.07 // Motor command that produces nominal Omega.

// If you are having trouble getting a good sensor signal, try adjusting
#define LEDDRIVE 1.0// Vary optical sensor LED brightness (0.7->1.0) 
#define TICKEDGE FALLING // Switching to RISING might improve noise.

// Use these offsets to avoid repeatedly typing values in plotter send window.  
// Offsets are ADDED to send window value (plus any potentiometer value).
#define offsetTicksPerUpdate 0 
#define offsetKff 0.0  // offset for feedforward gain
#define offsetKp 0.0   // offset for feedback gain.
#define offsetRepeats 0 // Stretches out the plot for each each sample point
#define offsetFreq 0.0  // offset input frequency
#define offsetAmp 0.0 // offset input amplitude
#define offsetTicksPerEstimate 0 
// ************************************************************************
// End of Constants and code for Users to Modify 
// SEE END OF FILE FOR USER-ALTERABLE CONTROL LOOP!!!


// ************************************************************************
// *************DO NOT CHANGE BELOW THIS LINE UNLESS CERTAIN!**************
// ************************************************************************

// Plotter Send window variable indexes. In serial plotter send window, 
// e.g. to set KP to 0.3, type "3 0.3" and click send, 
// e.g. to set the input period to two seconds, type "5 0.5" and click send.
// **********NEVER CHANGE THESE INDICES!  USE PLOTTER SEND WINDOW
// ********************NEVER CHANGE THESE INDICES! 
#define TICKSPERUPDATE    0     // 1->8, .
#define KFF               1     // 0.01->4   Feedforward or input gain
#define KI                2     // Unused 
#define KP                3     // 0.1->20   Feedback proportional gain
#define REPEATS           4     // 0,1,2,3   Sample repeats when plotting
#define FREQ              5     // 0.1->100  Input Oscillation freq 
#define AMP               6     // -10->10   Input Amp (if < 0, sine) 
#define TICKSPERESTIMATE  7     //  1->TICKSPERUPDATE
// **********NEVER CHANGE THESE INDICES!  USE PLOTTER SEND WINDOW
// **********NEVER CHANGE THESE INDICES!  USE PLOTTER SEND WINDOW


// SETTING UP THE REAL TIME INPUTS AND PLOTS
// ************************************************************************
// Lab Specific input and plot configurations are specified 
// with a configuration string,which specifies real time inputs and plots.

// The character "~" used as a delimiter between entries in the config string.
//
//  Plotter Send Box Input Definitions (aka real time adjustable values)
//  Format: "S~Variable Title~minVal~maxVal~".
//  Type "input# inputval" in serial plotter send box, then
//  call my6310.getSlider(slider#) in sketch to get sliderval.
//  Pound-defines above are handy to keep track of the slider order!!
String inputsRT = "S~Ticks/Update~0~1~S~Kff~0.1~10~S~Ki~0~100~S~Kp~0~10~S~Repeats~0~5~S~Freq~0.1~200~S~Amp~0~2~S~Ticks/Est~0~8~";

//  PLOT Format: "P~Title~yMin~yMax~xMax~#Traces~
//  NO MORE than four traces per plot (first is red, second blue,etc)
//  Sends data to plotter in same the order as the plots, 
//  Use pound-defines to keep track.
#if MODE == 0
String outPlots = "P~Sensor Zero 10xMotorCmd~0~1~500~3~";
#else
String outPlots = "P~Meas Des MotorCmd~-1.5~1.5~500~3~";
//String outPlots = "P~Meas Des~-1.5~1.5~500~2~P~MotorCmd~-1.5~1.5~500~1~";
#endif

// Plot String and Array indexes and strings for plot data.
#define PLOT_MS 0
#define PLOT_DS 1
#define PLOT_CMD 2

// Now assemble the inputs and plotter configuration strings;
String config_message = inputsRT + outPlots;

// Set up interrupt handler and speed calculation for optical speed encoder
// ************************************************************************
#define nOptoSensor IN5
volatile uint32_t tTick = 0;
volatile boolean setSpeedFlag = false;  // Flag set when spd calculated, 
elapsedMicros tTickClk = 0;       // cleared when used in loop.  
                                                                 
void iservRPS() {
  tTick = tTickClk;  // Read clock
  setSpeedFlag = true;
}

#define LEVELDELAY 10
#define BOUNCETIME 10
#define TIMEOUT 100000
#define ticksPerRotation 8// 12-spoked encoder.
float scaleRPS = 1.0e6/float(ticksPerRotation);

float getSpeed(int ticks2up, int ticks2est) {
  uint32_t deltas[ticksPerRotation]; 
  boolean gotOne = false;
  elapsedMicros tOutCnt = 0;

  // Set to appropriate values if zero.
  if(ticks2up == 0) ticks2up = ticksPerRotation;
  if(ticks2est == 0) ticks2est = ticks2up;
  
  // Get ticks2up good ticks, and leave flag ready for next time.
  for(int tickCnt=0; tickCnt < ticks2up; tickCnt++) {
    do {
      for(tOutCnt = 0; !setSpeedFlag && tOutCnt < TIMEOUT;) {}; // Wait for tick.
      if(tOutCnt >= TIMEOUT) {
        gotOne = true;
        tTick = tTickClk; 
      } else {
        delayMicroseconds(LEVELDELAY);
        boolean val = digitalReadFast(nOptoSensor);
        if(TICKEDGE == FALLING) val = !val;
        gotOne = val && (tTick > BOUNCETIME);
      } 
      if(gotOne) {  // True transition.
        deltas[tickCnt] = tTick;
        tTickClk -= tTick;
      }
      setSpeedFlag = false;
    } while(!gotOne);
  }

  // Sum deltas.
  uint32_t deltaSum = deltas[ticks2up - 1];
  for(int i = 2; i <= ticks2est; i++) deltaSum += deltas[ticks2up - i];
  // Compute speed and return.
  return(scaleRPS*float(ticks2est)/float(deltaSum));
}

// ************************************************************************
// Done with Set up of optical speed encoder


// Control loop setup
// ************************************************************************
#if MODE == 0
elapsedMicros loopTimer = 0;  // Times oscilloscope traces.
#endif

void setup() {

  // Setup control loop
  my6310.setup(config_message);
  my6310.setupADC(32,false); 

  // Set up the motor driver PWM
  my6310.startPWM(1.0);

  // Set up optical sensor input and driver for optical sensor's LED.
  pinMode(nOptoSensor, INPUT_PULLUP);
  my6310.pwmWrite(O3,LEDDRIVE); 
  my6310.pwmWrite(O4,0.0); 

  // Set up optical sensor interrupt routine.
  #if MODE == 1
    attachInterrupt(digitalPinToInterrupt(nOptoSensor), iservRPS, TICKEDGE);
  #endif

  // Delay before beginning.
  delay(5);
}

//********************** Main USER ALTERABLE Control Loop **********************
//*******************************************************************************

#define bufSize 500
float sensorVals[bufSize];

void loop() {  
  // Initializes, gets GUI updates.
  my6310.startloop();
  
  // Get nominal command
  float nomCmd = nomCommand + 0.03*(my6310.analog2param(IN8)-0.5);

  // Get control parameters.
#if MODE == 1
  int ticksPerUpIn = offsetTicksPerUpdate + my6310.getSlider(TICKSPERUPDATE);
  float Kff = offsetKff + my6310.getSlider(KFF); 
  float Kp = offsetKp + my6310.getSlider(KP); 
  int numSends = 1 + offsetRepeats + my6310.getSlider(REPEATS); 
  float ampIn = offsetAmp + my6310.getSlider(AMP);  
  float freqIn = offsetFreq + my6310.getSlider(FREQ); 
  int ticksPerEstIn = offsetTicksPerEstimate + my6310.getSlider(TICKSPERESTIMATE);
#endif

#if MODE == 0  // Oscilloscope mode, just plots sensor behavior.
  for(loopTimer = 0; int(loopTimer) < 200000;) {};   // Wait two seconds between samples.
  uint16_t sampleBuf = bufSize;
  my6310.adcGetBuf(nOptoSensor,sampleBuf,sensorVals); // get sensor data
  float motorCmd = nomCmd;
  int numSends = bufSize;
 
#else  // Control Mode
  // Compute the desired offset-from-nominal speed.
  float omega_d = my6310.updateDesired(ampIn,freqIn,0);

  // Measure the offset-from-nominal speed.
  float omega = getSpeed(ticksPerUpIn,ticksPerEstIn) - nomOmega;
 
  // Compute the offset-from-nominal motor command
  float cmd = Kp*(omega_d - omega) + Kff*omega_d; 

  // Offset the motor command from the nominal
  float motorCmd = nomCmd + (cmd/nomOmega);  // Normalizing by nomOmega should be absorbed in model!
#endif

  // Write command to motor, dCmd scaled by nominal Omega for convenience. 
  float cmdClip = my6310.hbridgeBipolar(motorCmd,B2L,B1L); // B2L<->B1L changes dir.
  my6310.hbridgeBipolar(motorCmd,A2L,A1L); // B2L<->B1L changes dir.

#if MODE == 1 // Only create disturbance during control mode.
  // Create disturbance, spin forward for first half cycle, backward for second.
  float sinceFlip = my6310.getInputPeriodFrac();
  float disturb = 0.0;
  if((sinceFlip > 0.15) && (sinceFlip < 0.35)) disturb = DisturbA; // Pick MOST disturbing direction!
  else if((sinceFlip > 0.65) && (sinceFlip < 0.85)) disturb = -DisturbA;
  my6310.hbridgeBipolar(disturb,A1R,A2R);   // Drive two outputs in case one is dead.
  my6310.hbridgeBipolar(disturb,B1R,B2R);
#endif 

  // Display results
  float loopStatus[3];
  for (int i = 0; i < numSends; i++) { // Send data to monitor.
  #if MODE == 0
    loopStatus[PLOT_MS] = sensorVals[i];
    loopStatus[PLOT_DS] = 0.0;
    loopStatus[PLOT_CMD] = 10*cmdClip;
  #else 
    loopStatus[PLOT_MS] = omega; // Plot Measured omega in RPS
    loopStatus[PLOT_DS] = omega_d;  // Plot Desired omega in RPS
    loopStatus[PLOT_CMD] = (cmdClip - nomCmd);
  #endif
    my6310.sendStatus(3,loopStatus); 
  } 

}



  


  
