/*
  Library for Teensy 4.1 6.310 board
  Created by teaching staff of 6.310, a control class at MIT.
*/

#ifndef lib6310_h
#define lib6310_h

#include <ADC.h>

// Dual Hbridge (A and B) with Crnt and Volt Sense
#define A1L 0
#define A2L 1
#define B1L 2
#define B2L 3

// Second Dual Hbridge (A_T and B_T) (no crnt sensing)
#define A1R 4
#define A2R 5
#define B1R 6
#define B2R 7

#define O1 22
#define O2 23
#define O3 15
#define O4 14

#define IN1 A7
#define IN2 A6
#define IN3 A5
#define IN4 A4
#define IN5 A3
#define IN6 A2
#define IN7 A17
#define IN8 A16

// Hbridge Crnt and Volt Sense Inputs 
#define IsenseAL A10
#define IsenseBL A11
#define IsenseAR A12
#define IsenseBR A13
#define hEnablePWM 29

// Loop monitor Pin (for testing)
#define loopMonitor 32

// End Pin definitions.

// Definitions for ADC
#define LIB6310_ADC(BUF)\
ADC *_adc = new ADC(); \
volatile uint16_t _bufMax = BUF;\
volatile uint16_t _numSamples0 = 0;\
volatile uint16_t _numSamples1 = 0;\
volatile uint16_t _adcArray0[BUF+10];\
volatile uint16_t _adcArray1[BUF+10];\
volatile uint16_t _numS0 = 0;\
volatile uint16_t _numS1 = 0;\

// Definitions for reading parameters from potentiometers
#define zeroParamThresh 0.005
#define scaleParam 1.005

#define MAXPLOT 10
class lib6310
{
 public:
  lib6310(int pwmRes, int adcRes, Stream *serialChan);
  void setup(String config_message);
  void startPWM(float fraction);
  float hbridgeWrite(float normV, int pin);
  float hbridgeBipolar(float normV, int pinP, int pinN);
  void sendStatus(int numStats, float stats[]);
  void startloop();
  bool isFirst();
  void setFirst(boolean val);
  float updateDesired(float Amp, float Freq, float smooth);
  float getSlider(int whichVal);
  float getAdcMax();
  float getDacMax();
  float getInputPeriodFrac();
  void setupADC(int numAvg, boolean goFast);
  float adcGetAvg(const uint8_t pin);
  float adcGetAvg(const uint8_t pin, uint16_t &numSamples);
  float adcGetBuf(const uint8_t pin, uint16_t &numSamples, float sampBuf[]);
  float pwmWrite(int pin, float normV);
  float analog2param(int pin);
  float scaleAndBound(float val, float Thresh, float Scale);
 private:
  void _processRcvString(String inStr, float rcvBuf[]);
  void _parseConfigStr(String inStr);
  float _adcReadIn(const uint8_t pin, const uint16_t numSamples);
  float _rcvBuf[10];
  char _sendBuf[200];
  int _dacRes;
  int _adcRes;
  float _dacMax;
  float _adcMax;
  float _PWMF;
  Stream *_serChan;
  int _baud;
  boolean _firstTime = true;
  boolean _firstStatus = true;
  boolean _monitorSwitch = false;
  String _config_message = "";
  String _inputString = "";
  int _numPlots = 0;
  float _plotOffsets[MAXPLOT];
  float _plotMins[MAXPLOT];
  float _plotMaxs[MAXPLOT];
  String _plotLabels[MAXPLOT];
  elapsedMicros _inputTimer;
  float _saveDesired;
  float _inputPeriodFrac;
};


class backEMF 
{
 public:
  backEMF(int pVa, int pVb, int pC, float cS, float vS, float Rm);
  float bEMF(int nAvg, lib6310 lib);
 private:
  int _pVa;
  int _pVb;
  int _pC;
  float _cScale;
  float _vScale;
  float _Rm;
};

//*************Matrix operations 
// Typedef for the three and four long vectors, and 3x3 and 4x4 matrices
typedef float vec3[3];
typedef vec3 mat3x3[3];
typedef float vec4[4];
typedef vec4 mat4x4[4];


//*************Matrix operations borrowed from linmath.h
static inline void mat3x3_mul_vec3(vec3 r, mat3x3 M, vec3 v)
{
  int i, j;
  for(j=0; j<3; ++j) {
    r[j] = 0.0;
    for(i=0; i<3; ++i)
      r[j] += M[j][i] * v[i]; // Edited to untranspose M
  }
}

static inline void copy_bound_vec3(vec3 r, vec3 a, float bound)
{
  int j;
  for(j=0; j<3; ++j) {
    if(a[j] > bound) r[j] = bound;
    else if(a[j] < -bound) r[j] = -bound;
    else r[j] = a[j];
  }
}

#define LINMATH_DEFINE_VEC(n) \
static inline void vec##n##_add(vec##n r, vec##n const a, vec##n const b) \
{ \
  int i; \
  for(i=0; i<n; ++i) \
    r[i] = a[i] + b[i]; \
} \
static inline void vec##n##_sub(vec##n r, vec##n const a, vec##n const b) \
{ \
  int i; \
  for(i=0; i<n; ++i) \
    r[i] = a[i] - b[i]; \
} \
static inline void vec##n##_scale(vec##n r, vec##n const v, float const s) \
{ \
  int i; \
  for(i=0; i<n; ++i) \
    r[i] = v[i] * s; \
} \
static inline float vec##n##_mul_inner(vec##n const a, vec##n const b) \
{ \
  float p = 0.; \
  int i; \
  for(i=0; i<n; ++i) \
    p += b[i]*a[i]; \
  return p; \
} 
LINMATH_DEFINE_VEC(3)
LINMATH_DEFINE_VEC(4)
//************END BORROWED FROM LINMATH.H



#endif
